import Gnuplot
from operator import add

class Histogram:

    def __init__(self, values=None, bin_size=None):
        if bin_size:
            self.bin_size = bin_size
        self.bins = {}
        if values:
            assert bin_size, \
                   'You have to pass a bin size ' \
                   'if you pass a list of values.'
            self.incorporate(values)

    def incorporate(self, values):
        assert None not in values # This could cause your X session to die!
        tmp = []
        values = values[:]
        values.sort()
        l = len (values)
        for v in values:
            bin = int(v / float(self.bin_size))
            tmp.append(bin)
            self.bins[bin] = self.bins.get(bin, 0) + 1
        for bin in range(min(self.bins.keys())-1, max(self.bins.keys())+2):
            self.bins[bin] = self.bins.get(bin, 0)

    def compute_points(self):
        self.points = []
        assert None not in self.bins.values()
        for bin, value in self.bins.items():
            self.points.append((bin*self.bin_size, value))
        self.points.sort()

    def plot(self, graph=None):
        if not graph:
            self.graph = graph = Gnuplot.Gnuplot()
        graph('set data style boxes')
        self.compute_points()
        self.graph.plot (self.points)

class Multihist:

    def __init__(self, bin_size=None):
        self.graph = Gnuplot.Gnuplot()
        self.graph('set linestyle 2 lt 2')
        self.graph('set linestyle 1 lt 1')
        self.graph('set data style boxes')
        self.bin_size = bin_size
        self.number_of_graphs = 0
        
    def plot(self, values, bin_size=None):
        if bin_size is None:
            bin_size = self.bin_size
        assert bin_size, 'You have to specify a bin size; no default given.'
        histogram = Histogram(values, bin_size)
        histogram.compute_points()
        translated_points = []
        offset = (bin_size/2) * self.number_of_graphs
        for x, y in histogram.points:
            translated_points.append((x + offset, y))
        if translated_points:
            plotitem = Gnuplot.Data(translated_points)
            linestyle =self.number_of_graphs + 1
            plotitem.set_option(with='boxes ls %i' % linestyle)
            self.graph.replot(plotitem)
            self.number_of_graphs = self.number_of_graphs + 1

    def multi_plot(self, value_lists, number_bins):
        lists = reduce(add, value_lists)
        
        # This could cause your X session to die!
        assert None not in lists

        max_val, min_val = max(lists), min(lists)
        bin_size = (max_val - min_val) / float(number_bins)
        for list_idx, value_list in iterate(value_lists):
            self.plot(value_list, bin_size)

    def xlabel(self, name):

        self.graph.xlabel(name)

    def ylabel(self, name):

        self.graph.ylabel(name)

    def hardcopy(self, filename):

        self.graph('set term postscript eps enhanced')
        self.graph.hardcopy(filename)

if __name__ == '__main__':

    l1 = [3, 5, 3.3]
    l2 = [20, 20.2, 35, 41]
    # e = Histogram(l1, 1)
    # raw_input()
    f = Multihist(1)
    f.plot(l1)
    f.plot(l2)
    f.hardcopy('/tmp/tst_histogram.ps')
