#include "Python.h"
#include "arrayobject.h"
#include "tools/Sequences/_sequence.h"
#include "RNA/MSARi/MSA/tuples/_msa.h"

/* Uses definitions from mouse.sequences._sequencesmodule.c */

/* Which nucleotides can be complements in a ncRNA */
unsigned int complements[4][4] = {
  {0, 0, 0, 1},  /* A */
  {0, 0, 1, 0},  /* C */
  {0, 1, 0, 1},  /* G */
  {1, 0, 1, 0}   /* T */    
};

/* List of nucleotide pairs that can be complements */
unsigned int comp_list[7][2] = {
  {DNA_A, DNA_T},
  {DNA_C, DNA_G},
  {DNA_G, DNA_C},
  {DNA_G, DNA_T},
  {DNA_T, DNA_A},
  {DNA_T, DNA_G}  
};

PyObject *get_column_stats(PyObject *self, PyObject *args) {

  char *col1, *col2;
  unsigned int len1, len2, compidx, colidx, numevents, numcomps;
  unsigned int nuke1, nuke2;
  float odds, *odds1, *odds2;
  PyArrayObject *_odds1, *_odds2;

  if (!PyArg_ParseTuple(args, "s#s#O!O!",
			&col1, &len1, &col2, &len2,
			&PyArray_Type, &_odds1,
			&PyArray_Type, &_odds2)) {
    return NULL;
  }

  if (len1 != len2) {
    PyErr_SetString(PyExc_ValueError,"Columns should be same height");
    return NULL;
  }

  if ((_odds1->nd != 1) || (_odds2->nd != 1)) {
    PyErr_SetString(PyExc_ValueError,"Arrays should be one-dimensional");
    return NULL;
  }

  if ((_odds1->dimensions[0] != 4) || (_odds2->dimensions[0] != 4)) {
    PyErr_SetString(PyExc_ValueError,"Arrays should have four elements");
    return NULL;
  }

  if ((_odds1->descr->type_num != PyArray_FLOAT) || \
      (_odds2->descr->type_num != PyArray_FLOAT)) {
    PyErr_SetString(PyExc_ValueError,"Arrays should be comprised of floats.");
    return NULL;
  }

  odds1 = (float *)(_odds1->data);
  odds2 = (float *)(_odds2->data);
  odds = 0;
  for (compidx = 0; compidx < 6; compidx++) {
    odds += odds1[comp_list[compidx][0]] * odds2[comp_list[compidx][1]];
  }

  numevents = numcomps = 0;
  for (colidx = 0; colidx < len1; colidx++) {
    nuke1 = _sequence_dna_char_num[(unsigned int)col1[colidx]];
    nuke2 = _sequence_dna_char_num[(unsigned int)col2[colidx]];
    if ((nuke1 != AA) && (nuke2 != AA)) {
      numevents++;
      if (complements[nuke1][nuke2]) {
	numcomps++;
      }
    }
  }

  return Py_BuildValue("iif", numcomps, numevents, odds);
}

PyObject *neighbour_odds(PyObject *self, PyObject *args) {

  PyArrayObject *pair_odds;
  PyObject *rv, *actual_rv, *odd_value;
  int pos1, pos2, o1, o2, length, p1start, p1end, p2start, p2end;
  int stride, number_cells;
  float odds;
  float significance_odds=0.05;

  if (!PyArg_ParseTuple(args, "O!ii",
			&PyArray_Type, &pair_odds, &pos1, &pos2)) {
    return NULL;
  }
  if ((pair_odds->nd != 2) || \
      ((length = pair_odds->dimensions[0]) != pair_odds->dimensions[1])) {
    PyErr_SetString(PyExc_ValueError, "odds should be a square array.");
    return NULL;
  }
  if (pair_odds->descr->type_num != PyArray_FLOAT) {
    PyErr_SetString(PyExc_ValueError, "odds should be a square array of floats.");
    return NULL;
  }
  pair_odds = (PyArrayObject *)
    PyArray_ContiguousFromObject((PyObject *)pair_odds,
				 PyArray_FLOAT, 2, 2);
  if ((pos1 < 0) || (pos1 >= length) || (pos2 < 0) || (pos2 >= length)) {
    PyErr_SetString(PyExc_ValueError, "pos1, pos2 should be an index into odds");
    return NULL;
  }
  if (pos1 >= pos2) {
    PyErr_SetString(PyExc_ValueError, "pos1 should precede pos2");
    return NULL;
  }
  if (!(rv = PyList_New(0))) {
      return NULL;
  }

  number_cells = 0;

  /* Find statistically significant pairs prior to pos1 and following
     pos2 */
  p1start = 10;
  if (pos1 < p1start) {
    p1start = pos1;
  }
  if ((length - 1 - pos2) < p1start) {
    p1start = length - 1 - pos2;
  }
  for (o1 = p1start; o1 > 0; o1--) {
    stride = length*(pos1 -o1);
    if (o1 == 1) {
      p2start = 0;
    } else {
      p2start = -1;
    }
    p2end = 1;
    for (o2 = p2start; o2 <= p2end; o2++) {
      odds = ((float *)(pair_odds->data))[stride + pos2 + o1 + o2];
      if (odds != -1) {
	number_cells++;
	if (odds < significance_odds) {
	  odd_value = Py_BuildValue("f", odds);
	  if (!odd_value) {
	    return NULL;
	  }
	  if (PyList_Append(rv, odd_value) == -1) {
	    return NULL;
	  }
	  Py_DECREF(odd_value);
	}
      }
    }
  }

  /* Now for the inner annealment */
  p1end = 10;
  if ((((pos2-pos1)/2) - 3) < p1end) {
    p1end = ((pos2-pos1)/2) - 3;
  }
  for (o1 = 1; o1 <= p1end; o1++) {
    stride = length*(pos1+o1);
    p2start = -1;
    if (o1 == 1) {
      p2end = 0;
    } else {
      p2end = 1;
    }
    for (o2 = p2start; o2 <= p2end ; o2++) {
      odds = ((float *)(pair_odds->data))[stride+pos2 - o1 +o2];
      if (odds != -1) {
	number_cells++;
	if (odds < significance_odds) {
	  odd_value = Py_BuildValue("f", odds);
	  if (!odd_value) {
	    return NULL;
	  }
	  if (PyList_Append(rv, odd_value) == -1) {
	    return NULL;
	  }
	  Py_DECREF(odd_value);
	}
      }
    }
  }
  if (!(actual_rv = Py_BuildValue("iO", number_cells, rv))) {
    return NULL;
  }
  Py_DECREF(rv); Py_DECREF(pair_odds);
  return actual_rv;
}

static PyMethodDef _msaMethods[] = {
  {"get_column_stats", get_column_stats, METH_VARARGS},
  {"neighbour_odds",   neighbour_odds,   METH_VARARGS},
  {NULL, NULL}
};

void init_msa(void) {
  (void)Py_InitModule("_msa", _msaMethods);
  import_array();
}
