#include <fstream.h>
#include <iostream.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "fseq.h"
#include "splice.h"

#define DEBUGFIND false
#define DEBUGTEST false

// Splice functions

Splice::Splice(bool flag) {

  acceptor = flag;

  if (acceptor) {
    
    for (int i=0; i < 41; i++) {
      singles[i] = new QProb();
      Nsingles[i] = new QProb();
      pairs[i] = new QuadProb();
      Npairs[i] = new QuadProb();
    }
    for (int i=0; i < 23; i++) {
      branchnormal[i] = new QuadProb();
      Nbranchnormal[i] = new QuadProb();
    }
    for (int i=0; i < 18; i++) {
      branchpoint[i] = new Quad4Prob();
      Nbranchpoint[i] = new Quad4Prob();
    }
  }

  // otherwise it's a donor splice site
  else {

    initTables();
    for (int i=0; i < 6; i++) 
      NH5table[i] = new QProb();
    for (int i=0; i < 5; i++) 
      NHM1table[i] = new QProb();
    for (int i=0; i < 4; i++) 
      NHM2table[i] = new QProb();
    for (int i=0; i < 3; i++) 
      NH6atable[i] = new QProb();
    for (int i=0; i < 3; i++) 
      NH6btable[i] = new QProb();

    Ntotcount = NH5count = NHM1count = NHM2count = 
      NH6count = 0.0; 
  }
  
  maxsites = 1000;
  numsites = 0;
  sites = new (Site*)[maxsites];
}

Splice::~Splice() {
  
  if (acceptor) {
    
    for (int i=0; i < 41; i++) {
      delete singles[i];
      delete Nsingles[i];
      delete pairs[i];
      delete Npairs[i];
    }
    for (int i=0; i < 23; i++) {
      delete branchnormal[i];
      delete Nbranchnormal[i];
    }
    for (int i=0; i < 18; i++) {
      delete branchpoint[i];
      delete Nbranchpoint[i];
    }
  }

  else {

    for (int i=0; i < 6; i++) 
      delete NH5table[i];
    for (int i=0; i < 5; i++) 
      delete NHM1table[i];
    for (int i=0; i < 4; i++) 
      delete NHM2table[i];
    for (int i=0; i < 3; i++) 
      delete NH6atable[i];
    for (int i=0; i < 3; i++) 
      delete NH6btable[i];
  }
  
  for (int i=0; i < numsites; i++)
    delete sites[i];
  delete sites;
}


void Splice::findSplices(GeneSequence *seq) {

  if (acceptor) 
    findAcceptorSplices(seq);
  else 
    findDonorSplices(seq);
}


void Splice::findAcceptorSplices(GeneSequence *seq) {
  
  // search through the sequence for AG pairs
  long length = seq->get_length() - 4;  // don't need to look at last 4

  // don't need to start looking until 41st base
  for (long i=41; i < length; i++) {

    // if the first bases match, let's compute...
    if (seq->get(i) == BASE_A) {
      if (seq->get(i+1) == BASE_G) {

	// if it's a true splice site:
	if (seq->whatRegion(i) == REGION_INTRON && 
	    seq->whatRegion(i+2) == REGION_CEXON) {

	  if (DEBUGFIND)
	    cout << "TRUE: basepair #" << i << endl;

	  Nucleotide previous_base = seq->get(i-37); 
	  int counter = 0;
	  
	  for (long index = i - 36; index < i + 5; index++) {
	    
	    Nucleotide base = seq->get(index);
	    // singles:
	    singles[counter]->addToCount(base);
	    // pairs:
	    pairs[counter]->addToCount(previous_base, base);
	    // branchpoints:
	    if (counter < 18) { // branchpoint region
	      branchpoint[counter]->addToCount(seq->get(index-4), 
					      seq->get(index-3),
					      seq->get(index-2));
	      branchpoint[counter]->addToCount(seq->get(index-3), 
					      seq->get(index-2),
					      previous_base);
	      branchpoint[counter]->addToCount(seq->get(index-2), 
					       previous_base, base);
	      branchpoint[counter]->addToCount(previous_base, base, 
					      seq->get(index+1));
	      branchpoint[counter]->addToCount(base, seq->get(index+1),
					      seq->get(index+2));
	    }
	    else
	      branchnormal[counter-18]->addToCount(previous_base, base);
	    
	    previous_base = base;
	    counter++;
	  }
	}
	 
	// else it's a pseudo one...
	else {

	  if (DEBUGFIND)
	    cout << "PSEUDO: basepair #" << i << endl;
	  
	  Nucleotide previous_base = seq->get(i-37); 
	  int counter = 0;
	  
	  for (long index = i - 36; index < i + 5; index++) {
	    
	    Nucleotide base = seq->get(index);
	    // singles:
	    Nsingles[counter]->addToCount(base);
	    // pairs:
	    Npairs[counter]->addToCount(previous_base, base);
	    // branchpoints:
	    if (counter < 18) { // branchpoint region
	      Nbranchpoint[counter]->addToCount(seq->get(index-4), 
						seq->get(index-3),
						seq->get(index-2));
	      Nbranchpoint[counter]->addToCount(seq->get(index-3), 
						seq->get(index-2),
						previous_base);
	      Nbranchpoint[counter]->addToCount(seq->get(index-2), 
						previous_base, base);
	      Nbranchpoint[counter]->addToCount(previous_base, base, 
						seq->get(index+1));
	      Nbranchpoint[counter]->addToCount(base, seq->get(index+1),
						seq->get(index+2));
	    }
	    else
	      Nbranchnormal[counter-18]->addToCount(previous_base, base);
	    
	    previous_base = base;
	    counter++;
	  }
	}
      }
    }
  }
}


void Splice::findDonorSplices(GeneSequence *seq) {
  
  // search through the sequence for GT pairs
  long length = seq->get_length() - 5;  // don't need to look at last 5
  
  // don't need to start looking until 4th base
  for (long i=4; i < length; i++) {
    
    // if the first bases match, let's compute...
    if (seq->get(i) == BASE_G) {
      if (seq->get(i+1) == BASE_T) {
	
	// if it's a true splice site:
	if (seq->whatRegion(i) == REGION_INTRON && 
	    seq->whatRegion(i-1) == REGION_CEXON) {
	  ;
	}	 
	// else it's a pseudo one...
	else {

	  Ntotcount++;
	  Nucleotide bpM3 = seq->get(i-3);  
	  Nucleotide bpM2 = seq->get(i-2);
	  Nucleotide bpM1 = seq->get(i-1);
	  Nucleotide bp3 = seq->get(i+2);
	  Nucleotide bp4 = seq->get(i+3);
	  Nucleotide bp6 = seq->get(i+5);

	  if (seq->get(i+4) != BASE_G) {
	    
	    NH5table[0]->addToCount(bpM3);
	    NH5table[1]->addToCount(bpM2);
	    NH5table[2]->addToCount(bpM1);
	    NH5table[3]->addToCount(bp3);
	    NH5table[4]->addToCount(bp4);
	    NH5table[5]->addToCount(bp6);
	  }
	  
	  // else bp 5 == G
	  else {
	    
	    NH5count++;
	    if (bpM1 != BASE_G) {
	      
	      NHM1table[0]->addToCount(bpM3);
	      NHM1table[1]->addToCount(bpM2);
	      NHM1table[2]->addToCount(bp3);
	      NHM1table[3]->addToCount(bp4);
	      NHM1table[4]->addToCount(bp6);
	    }
	    
	    // else bp -1 == G
	    else {
	      
	      NHM1count++;
	      if (bpM2 != BASE_A) {
		
		NHM2table[0]->addToCount(bpM3);
		NHM2table[1]->addToCount(bp3);
		NHM2table[2]->addToCount(bp4);
		NHM2table[3]->addToCount(bp6);
	      }
	      
	      // else bp -2 == A
	      else {
		
		NHM2count++;
		if (bp6 != BASE_T) {
		  
		  NH6atable[0]->addToCount(bpM3);
		  NH6atable[1]->addToCount(bp3);
		  NH6atable[2]->addToCount(bp4);
		}
		
		// else bp 6 == T
		else {
		  
		  NH6count++;
		  NH6btable[0]->addToCount(bpM3);
		  NH6btable[1]->addToCount(bp3);
		  NH6btable[2]->addToCount(bp4);
		}
	      }
	    }
	  }
	}
      }
    }
  }
}


istream& operator>>(istream& cin, Splice& splice) {

  cin >> splice.acceptor;
  
  if (splice.acceptor) {
    
    for (int i=0; i < 41; i++) 
      cin >> *(splice.singles[i]);
    for (int i=0; i < 41; i++) 
      cin >> *(splice.Nsingles[i]);
    for (int i=0; i < 41; i++) 
      cin >> *(splice.pairs[i]);
    for (int i=0; i < 41; i++) 
      cin >> *(splice.Npairs[i]); 
    for (int i=0; i < 23; i++) 
      cin >> *(splice.branchnormal[i]);
    for (int i=0; i < 23; i++) 
      cin >> *(splice.Nbranchnormal[i]);
    for (int i=0; i < 18; i++) 
      cin >> *(splice.branchpoint[i]);
    for (int i=0; i < 18; i++) 
      cin >> *(splice.Nbranchpoint[i]);
  }
  
  else {
    
    for (int i=0; i < 6; i++) 
      for (int j=0; j < 4; j++) 
	cin >> splice.H5table[i][j];
    for (int i=0; i < 5; i++) 
      for (int j=0; j < 4; j++) 
	cin >> splice.HM1table[i][j];
    for (int i=0; i < 4; i++) 
      for (int j=0; j < 4; j++) 
	cin >> splice.HM2table[i][j];
    for (int i=0; i < 3; i++) 
      for (int j=0; j < 4; j++) 
	cin >> splice.H6atable[i][j];
    for (int i=0; i < 3; i++) 
      for (int j=0; j < 4; j++) 
	cin >> splice.H6btable[i][j];
    for (int i=0; i < 6; i++) 
      cin >> *(splice.NH5table[i]);
    for (int i=0; i < 5; i++) 
      cin >> *(splice.NHM1table[i]);
    for (int i=0; i < 4; i++) 
      cin >> *(splice.NHM2table[i]);
    for (int i=0; i < 3; i++) 
      cin >> *(splice.NH6atable[i]);
    for (int i=0; i < 3; i++) 
      cin >> *(splice.NH6btable[i]);
    cin >> splice.Ntotcount >> splice.NH5count 
	>> splice.NHM1count >> splice.NHM2count 
        >> splice.NH6count;
  }
  
  return cin;
}

ostream& operator<<(ostream& cout, Splice& splice) {

  cout << splice.acceptor << endl;

  if (splice.acceptor) {
    
    for (int i=0; i < 41; i++) 
      cout << *(splice.singles[i]) << endl;
    for (int i=0; i < 41; i++) 
      cout << *(splice.Nsingles[i]) << endl;
    for (int i=0; i < 41; i++) 
      cout << *(splice.pairs[i]) << endl;
    for (int i=0; i < 41; i++) 
      cout << *(splice.Npairs[i]) << endl;
    for (int i=0; i < 23; i++) 
      cout << *(splice.branchnormal[i]) << endl;
    for (int i=0; i < 23; i++) 
      cout << *(splice.Nbranchnormal[i]) << endl;
    for (int i=0; i < 18; i++) 
      cout << *(splice.branchpoint[i]) << endl;
    for (int i=0; i < 18; i++) 
      cout << *(splice.Nbranchpoint[i]) << endl;
  }
  
  else {

    for (int i=0; i < 6; i++) {
      for (int j=0; j < 4; j++) 
	cout << splice.H5table[i][j] << "\t";
      cout << "\n";
    }
    for (int i=0; i < 5; i++) {
      for (int j=0; j < 4; j++) 
	cout << splice.HM1table[i][j] << "\t";
      cout << "\n";
    }
    for (int i=0; i < 4; i++) {
      for (int j=0; j < 4; j++) 
	cout << splice.HM2table[i][j] << "\t";
      cout << "\n";
    }
    for (int i=0; i < 3; i++) {
      for (int j=0; j < 4; j++) 
	cout << splice.H6atable[i][j] << "\t";
      cout << "\n";
    }
    for (int i=0; i < 3; i++) {
      for (int j=0; j < 4; j++) 
	cout << splice.H6btable[i][j] << "\t";
      cout << "\n";
    }
    for (int i=0; i < 6; i++) 
      cout << *(splice.NH5table[i]) << endl;
    for (int i=0; i < 5; i++) 
      cout << *(splice.NHM1table[i]) << endl;
    for (int i=0; i < 4; i++) 
      cout << *(splice.NHM2table[i]) << endl;
    for (int i=0; i < 3; i++) 
      cout << *(splice.NH6atable[i]) << endl;
    for (int i=0; i < 3; i++) 
      cout << *(splice.NH6btable[i]) << endl;
    cout << splice.Ntotcount << "\t" << splice.NH5count 
	 << "\t" << splice.NHM1count << "\t" << splice.NHM2count 
	 << "\t" << splice.NH6count << endl;
  }
  
  return cout;
}


void Splice::computations() {
  
  if (acceptor) {
    
    for (int i=0; i < 41; i++) {
      singles[i]->computeProb();
      Nsingles[i]->computeProb();
      pairs[i]->computeProb();
      Npairs[i]->computeProb();
    }
    for (int i=0; i < 23; i++) {
      branchnormal[i]->computeProb();
      Nbranchnormal[i]->computeProb();
    }
    for (int i=0; i < 18; i++) {
      branchpoint[i]->computeProb();
      Nbranchpoint[i]->computeProb();
    }
    
    if (DEBUGFIND) {
      
      for (int i=0; i < 41; i++) {
	
	cout << "\tSingles:\nCount:"; 
	for (int j=0; j < 4; j++) 
	  cout << "\t" << Sequence::base2char((Nucleotide)j) << ": " 
	       << singles[i]->getCount((Nucleotide)j);
	cout << "\nProbs:"; 
	for (int j=0; j < 4; j++) 
	  cout << "\t" << Sequence::base2char((Nucleotide)j) << ": " 
	       << singles[i]->getProb((Nucleotide)j);
	cout << "\n\tNSingles:\nCount:"; 
	for (int j=0; j < 4; j++) 
	  cout << "\t" << Sequence::base2char((Nucleotide)j) << ": " 
	       << Nsingles[i]->getCount((Nucleotide)j);
	cout << "\nProbs:"; 
	for (int j=0; j < 4; j++) 
	  cout << "\t" << Sequence::base2char((Nucleotide)j) << ": " 
	       << Nsingles[i]->getProb((Nucleotide)j);
	cout << "\n"; 
      }
    }
  }

  else {
    
    for (int i=0; i < 6; i++) 
      NH5table[i]->computeProb();
    for (int i=0; i < 5; i++) 
      NHM1table[i]->computeProb();
    for (int i=0; i < 4; i++) 
      NHM2table[i]->computeProb();
    for (int i=0; i < 3; i++) 
      NH6atable[i]->computeProb();
    for (int i=0; i < 3; i++) 
      NH6btable[i]->computeProb();
  }      
}


void Splice::testSplices(GeneSequence *seq, ostream& outfile) {
  
  if (acceptor) 
    testAcceptorSplices(seq, outfile);
  else 
    testDonorSplices(seq, outfile);
}


void Splice::testAcceptorSplices(GeneSequence *seq, ostream& outfile) {
  
  static int lookedAt = 0;
  
  // search through the sequence for AG pairs
  long length = seq->get_length() - 4;  // don't need to look at last 4
  
  // don't need to start looking until 41st base
  for (long i=41; i < length; i++) {
    
    // if the first bases match, let's compute...
    if (seq->get(i) == BASE_A) {
      if (seq->get(i+1) == BASE_G) {
	
	double sprob = 1.0, pprob = 1.0, bprob= 1.0;
	double nsprob = 1.0, npprob = 1.0, nbprob= 1.0;
	int counter = 0;
	Nucleotide previous_base = seq->get(i-37);

	for (long index = i - 36; index < i + 5; index++) {
	    
	  Nucleotide base = seq->get(index);

	  sprob *= singles[counter]->getProb(base);
	  nsprob *= Nsingles[counter]->getProb(base);

	  // first pairs entry isn't really a pair -- it's a single
	  if (counter == 0) {
	    
	    double temp = 0.0, ntemp = 0.0;
	    for (int j=0; j < 4; j++) {
	      temp += pairs[0]->getProb((Nucleotide)j, base);
	      ntemp += Npairs[0]->getProb((Nucleotide)j, base);
	    }

	    pprob = temp;
	    npprob = ntemp;
	  }
	  else {
	    pprob *= pairs[counter]->getProb(previous_base, base);
	    npprob *= Npairs[counter]->getProb(previous_base, base);
	  }

	  // *** DO WE DO THE SAME THING HERE AS WE DID FOR THE FIRST PAIRS
	  // ***  ENTRY???  MAYBE WE SHOULD....
	  if (counter < 18) { // branchpoint region
	    Nucleotide otherbase = seq->get(index-2);
	    bprob *= branchpoint[counter]->getProb(otherbase, previous_base,
						   base);
	    nbprob *= Nbranchpoint[counter]->getProb(otherbase, previous_base,
						     base);
	  }
	  else {
	    bprob *= branchnormal[counter-18]->getProb(previous_base, base);
	    nbprob *= Nbranchnormal[counter-18]->getProb(previous_base, base);
	  }

	  counter++;
	  previous_base = base;
	}

#if 0
	// create a Site object

	// make sure we haven't maxed out
	if (numsites == maxsites) {
	  
	  int newmax = 2 * maxsites;
	  Site **newsites = new (Site*)[newmax];
	  
	  for (int j=0; j < numsites; j++) 
	    newsites[j] = sites[j];
	  delete[] sites;
	  sites = newsites;
	  maxsites = newmax;
	}
#endif

	double Sprob = log(sprob / nsprob);
	double Pprob = log(pprob / npprob);
	double Bprob = log(bprob / nbprob);

	char *locus = new char[500];
	strcpy(locus, seq->get_locus());

	bool trueSite = false;
	if (seq->whatRegion(i) == REGION_INTRON && 
	    seq->whatRegion(i+2) == REGION_CEXON) 
	  trueSite = true;
	
	if (Sprob < -5.0 && Pprob < -5.0 && Bprob < -5.0) {
	  ;
	}
	else {
	  outfile << trueSite << "\t" << Sprob << "\t" << Pprob << "\t" << 
	    Bprob << "\t" << (int)seq->get(i-1) << "\t" << (int)seq->get(i+2) 
		  << "\t" << lookedAt << "\t" << i << endl; 
	}
	
#if 0
        sites[numsites++] = new Site(Sprob, Pprob, Bprob, locus, i, trueSite, 
				     lookedAt);
	
	if (DEBUGTEST) {
	  cout << "\tSite:\nLocus: " << sites[numsites-1]->locus << 
	    "\tBasepair: " << sites[numsites-1]->basepair << "\nSingle: " << 
	    sites[numsites-1]->singleprob << "\tPair: " << 
	    sites[numsites-1]->pairprob << "\tBranch: " << 
	    sites[numsites-1]->branchprob << endl;
	}
#endif
	
      }
    }
  }
  
  lookedAt++;
}


void Splice::testDonorSplices(GeneSequence *seq, ostream& outfile) {
  
  static int lookedAt = 0;

  // search through the sequence for GT pairs
  long length = seq->get_length() - 5;  // don't need to look at last 5
  
  // don't need to start looking until 4th base
  for (long i=4; i < length; i++) {
    
    // if the first bases match, let's compute...
    if (seq->get(i) == BASE_G) {
      if (seq->get(i+1) == BASE_T) {
	
	// if it's a true splice site:
	bool trueSite = false;
	if (seq->whatRegion(i) == REGION_INTRON && 
	    seq->whatRegion(i-1) == REGION_CEXON) {
	  trueSite = true;
	}	 
	
	double probability, badprob = 1.0;
	Nucleotide bpM3 = seq->get(i-3);  
	Nucleotide bpM2 = seq->get(i-2);
	Nucleotide bpM1 = seq->get(i-1);
	Nucleotide bp3 = seq->get(i+2);
	Nucleotide bp4 = seq->get(i+3);
	Nucleotide bp6 = seq->get(i+5);
	
	
	if (seq->get(i+4) != BASE_G) {
	  
	  probability = 197.0 / 1254.0;
	  probability *= H5table[0][bpM3];
	  probability *= H5table[1][bpM2];
	  probability *= H5table[2][bpM1];
	  probability *= H5table[3][bp3];
	  probability *= H5table[4][bp4];
	  probability *= H5table[5][bp6];
	  badprob = (Ntotcount - NH5count) / Ntotcount;
	  badprob *= NH5table[0]->getProb(bpM3);
	  badprob *= NH5table[1]->getProb(bpM2);
	  badprob *= NH5table[2]->getProb(bpM1);
	  badprob *= NH5table[3]->getProb(bp3);
	  badprob *= NH5table[4]->getProb(bp4);
	  badprob *= NH5table[5]->getProb(bp6);
	}
	
	// else bp 5 == G
	else {
	  
	  probability = 1057.0 / 1254.0;
	  badprob = NH5count / Ntotcount;
	  
	  if (bpM1 != BASE_G) {
	    
	    probability *= 234.0 / 1057.0;
	    probability *= HM1table[0][bpM3];
	    probability *= HM1table[1][bpM2];
	    probability *= HM1table[2][bp3];
	    probability *= HM1table[3][bp4];
	    probability *= HM1table[4][bp6];
	    badprob *= (NH5count - NHM1count) / NH5count;
	    badprob *= NHM1table[0]->getProb(bpM3);
	    badprob *= NHM1table[1]->getProb(bpM2);
	    badprob *= NHM1table[2]->getProb(bp3);
	    badprob *= NHM1table[3]->getProb(bp4);
	    badprob *= NHM1table[4]->getProb(bp6);
	  }
	  
	  // else bp -1 == G
	  else {
	    
	    probability *= 823.0 / 1057.0;
	    badprob *= NHM1count / NH5count;

	    if (bpM2 != BASE_A) {
	      
	      probability *= 336.0 / 823.0;
	      probability *= HM2table[0][bpM3];
	      probability *= HM2table[1][bp3];
	      probability *= HM2table[2][bp4];
	      probability *= HM2table[3][bp6];
	      badprob *= (NHM1count - NHM2count) / NHM1count;
	      badprob *= NHM2table[0]->getProb(bpM3);
	      badprob *= NHM2table[1]->getProb(bp3);
	      badprob *= NHM2table[2]->getProb(bp4);
	      badprob *= NHM2table[3]->getProb(bp6);
	    }

	    // else bp -2 == A
	    else {
	      
	      probability *= 487.0 / 823.0;
	      badprob *= NHM2count / NHM1count;
	    
	      if (bp6 != BASE_T) {
	    
		probability *= 310.0 / 487.0;
		probability *= H6atable[0][bpM3];
		probability *= H6atable[1][bp3];
		probability *= H6atable[2][bp4];
		badprob *= (NHM2count - NH6count) / NHM2count;
		badprob *= NH6atable[0]->getProb(bpM3);
		badprob *= NH6atable[1]->getProb(bp3);
		badprob *= NH6atable[2]->getProb(bp4);
	      }
	      
	      // else bp 6 == T
	      else {
	      
		probability *= 177.0 / 487.0;
		probability *= H6btable[0][bpM3];
		probability *= H6btable[1][bp3];
		probability *= H6btable[2][bp4];
		badprob *= NH6count / NHM2count;
		badprob *= NH6btable[0]->getProb(bpM3);
		badprob *= NH6btable[1]->getProb(bp3);
		badprob *= NH6btable[2]->getProb(bp4);
	      }
	    }
	  }
	}
	
	double otherbadprob = .25 * .25 * .25 * .25 * .25 * .25 * .25;
	
	double answer = log(probability / badprob); 
	double otheranswer = log(probability / otherbadprob); 
	
	//	outfile << trueSite << "\t" << answer << "\t" << lookedAt 
	outfile << trueSite << "\t" << otheranswer << "\t" << answer << "\t" 
		<< lookedAt << "\t" << i << endl;
      }
    }
  }
  
  lookedAt++;
}


void Splice::initTables() {

  H5table[0][0] = 35.0 / 101.0;  H5table[0][1] = 44.0 / 101.0;  
  H5table[0][2] = 16.0 / 101.0;  H5table[0][3] = 6.0 / 101.0;  
  H5table[1][0] = 85.0 / 101.0;  H5table[1][1] = 4.0 / 101.0;  
  H5table[1][2] = 7.0 / 101.0;  H5table[1][3] = 5.0 / 101.0;  
  H5table[2][0] = 0.0195;  H5table[2][1] = 0.0095;  
  H5table[2][2] = .9695;  H5table[2][3] = 0.0015;  
  H5table[3][0] = 81.0 / 101.0;  H5table[3][1] = 3.0 / 101.0;  
  H5table[3][2] = 15.0 / 101.0;  H5table[3][3] = 2.0 / 101.0;  
  H5table[4][0] = 0.51;  H5table[4][1] = .28;  
  H5table[4][2] = 0.09;  H5table[4][3] = 0.12;  
  H5table[5][0] = 0.22;  H5table[5][1] = 0.2;  
  H5table[5][2] = 0.3;  H5table[5][3] = 0.28;  
  
  HM1table[0][0] = 29.0 / 99.0;  HM1table[0][1] = 31.0 / 99.0;  
  HM1table[0][2] = 21.0 / 99.0;  HM1table[0][3] = 18.0 / 99.0;  
  HM1table[1][0] = 43.0 / 101.0;  HM1table[1][1] = 30.0 / 101.0;  
  HM1table[1][2] = 17.0 / 101.0;  HM1table[1][3] = 11.0 / 101.0;  
  HM1table[2][0] = 55.95 / 99.0;  HM1table[2][1] = 0.05 / 99.0;  
  HM1table[2][2] = 42.95 / 99.0;  HM1table[2][3] = 0.05 / 99.0;  
  HM1table[3][0] = 93.0 / 101.0;  HM1table[3][1] = 2.0 / 101.0;  
  HM1table[3][2] = 3.0 / 101.0;  HM1table[3][3] = 3.0 / 101.0;  
  HM1table[4][0] = 5.0 / 101.0;  HM1table[4][1] = 10.0 / 101.0;  
  HM1table[4][2] = 10.0 / 101.0;  HM1table[4][3] = 76.0 / 101.0;  

  HM2table[0][0] = 0.29;  HM2table[0][1] = 0.30;  
  HM2table[0][2] = 0.18;  HM2table[0][3] = 0.23;  
  HM2table[1][0] = 0.42;  HM2table[1][1] = 0.01;
  HM2table[1][2] = 0.56;  HM2table[1][3] = 0.01;
  HM2table[2][0] = 0.80;  HM2table[2][1] = 0.04;
  HM2table[2][2] = 0.08;  HM2table[2][3] = 0.08;
  HM2table[3][0] = 0.14;  HM2table[3][1] = 0.21;
  HM2table[3][2] = 0.16;  HM2table[3][3] = 0.49;

  H6atable[0][0] = 39.0 / 99.0;  H6atable[0][1] = 43.0 / 99.0;  
  H6atable[0][2] = 15.0 / 99.0;  H6atable[0][3] = 2.0 / 99.0;  
  H6atable[1][0] = 46.0 / 101.0;  H6atable[1][1] = 6.0 / 101.0;  
  H6atable[1][2] = 46.0 / 101.0;  H6atable[1][3] = 3.0 / 101.0;  
  H6atable[2][0] = 69.0 / 101.0;  H6atable[2][1] = 5.0 / 101.0;  
  H6atable[2][2] = 20.0 / 101.0;  H6atable[2][3] = 7.0 / 101.0;  

  H6btable[0][0] = 0.32;  H6btable[0][1] = 0.40;  
  H6btable[0][2] = 0.23;  H6btable[0][3] = 0.05;  
  H6btable[1][0] = 0.27;  H6btable[1][1] = 0.04;
  H6btable[1][2] = 0.59;  H6btable[1][3] = 0.10;
  H6btable[2][0] = 0.51;  H6btable[2][1] = 0.05;
  H6btable[2][2] = 0.25;  H6btable[2][3] = 0.19;
}

