#include <string.h>
#include <stdlib.h>
#include <iostream.h>
#include "quads.h"

// QProb functions

QProb::QProb() {

  // initializations
  totalcount = 0;
  for (int i=0; i < 4; i++) {
    probs[i] = 0.0;      
    counts[i] = 0.0;      
  }
}


void QProb::addToCount(Nucleotide i) {
  
  // increase the total count
  totalcount++;
  
  // increase an individual count provided it's a known base
  if (i == BASE_UNKNOWN)
    i = BASE_C;  
  counts[i]++;
}


void QProb::computeProb() {
  
  // we cannot allow probability = 0....
  // first, find out how many zeros there are
  int zeros = 0;
      
  for (int i=0; i < 4; i++) 
    if (counts[i] == 0)
      zeros++;
  
  // skip if there are no zeros
  if (zeros > 0) {
    
    if (zeros == 4)
      cerr << "All zeros!" << endl;

    // otherwise, adjust probabilities accordingly
    double adjustment = (double)zeros / (double)(4 - zeros);
    
    for (int i=0; i < 4; i++) {
      if (counts[i] == 0)
	counts[i] = 1;
      else 
	counts[i] -= adjustment;
    }
  }
  
  // now, for some basic math....
  for (int i=0; i < 4; i++) 
    probs[i] = counts[i] / totalcount;
}


double QProb::getProb(Nucleotide i) {

  if (i == BASE_UNKNOWN)
    i = BASE_C;  
  return probs[i];
}


double QProb::getCount(Nucleotide i) {
  
  if (i == BASE_UNKNOWN)
    i = BASE_C;  
  return counts[i];
}


istream& operator>>(istream& cin, QProb& q) {

  cin >> q.totalcount;
  for (int i=0; i < 4; i++) 
    cin >> q.counts[i];
  for (int i=0; i < 4; i++) 
    cin >> q.probs[i];

  return cin;
}


ostream& operator<<(ostream& cout, QProb& q) {

  cout << q.totalcount << "\t";
  for (int i=0; i < 4; i++) 
    cout << q.counts[i] << "\t";
  cout << "\n";
  for (int i=0; i < 4; i++) 
    cout << q.probs[i] << "\t";
  cout << "\n";
  
  return cout;
}



// QuadProb functions

QuadProb::QuadProb() {

  // initializations
  totalcount = 0;
  for (int i=0; i < 4; i++) {
    for (int j=0; j < 4; j++) {
      probs[i][j] = 0.0;      
      counts[i][j] = 0.0;      
    }    
  }
}


void QuadProb::addToCount(Nucleotide before_i, Nucleotide i) {
  
  // increase the total count
  totalcount++;
  
  // increase an individual count provided it's a known base
  if (i == BASE_UNKNOWN)  
    i = BASE_C;    
  if (before_i == BASE_UNKNOWN)
    before_i = BASE_C;
  
  counts[before_i][i]++;
}


void QuadProb::computeProb() {
  
  // we cannot allow probability = 0....
  // first, find out how many zeros there are
  int zeros = 0;
      
  for (int i=0; i < 4; i++) 
    for (int j=0; j < 4; j++) 
      if (counts[i][j] == 0)
	zeros++;
  
  // skip if there are no zeros
  if (zeros > 0) {
    
    if (zeros == 16)
      cerr << "All zeros!" << endl;

    // otherwise, adjust probabilities accordingly
    double adjustment = (double)zeros / (double)(16 - zeros);
    
    for (int i=0; i < 4; i++) {
      for (int j=0; j < 4; j++) { 
	if (counts[i][j] == 0)
	  counts[i][j] = 1;
	else 
	  counts[i][j] -= adjustment;
      }
    }
  }
  
  // now, for some basic math....
  for (int i=0; i < 4; i++) 
    for (int j=0; j < 4; j++) 
      probs[i][j] = counts[i][j] / totalcount;
}


double QuadProb::getProb(Nucleotide before_i, Nucleotide i) {

  if (i == BASE_UNKNOWN)  
    i = BASE_C;    
  if (before_i == BASE_UNKNOWN)
    before_i = BASE_C;
  
  return probs[before_i][i];
}


double QuadProb::getCount(Nucleotide before_i, Nucleotide i) {

  if (i == BASE_UNKNOWN)  
    i = BASE_C;    
  if (before_i == BASE_UNKNOWN)
    before_i = BASE_C;

  return counts[before_i][i];
}


istream& operator>>(istream& cin, QuadProb& q) {
  
  cin >> q.totalcount;
  for (int i=0; i < 4; i++) 
    for (int j=0; j < 4; j++) 
      cin >> q.counts[i][j];
  for (int i=0; i < 4; i++) 
    for (int j=0; j < 4; j++) 
      cin >> q.probs[i][j];
  
  return cin;
}


ostream& operator<<(ostream& cout, QuadProb& q) {
  
  cout << q.totalcount << "\t";
  for (int i=0; i < 4; i++) {
    for (int j=0; j < 4; j++) 
      cout << q.counts[i][j] << "\t";
    cout << "\n";
  }
  for (int i=0; i < 4; i++) {
    for (int j=0; j < 4; j++) 
      cout << q.probs[i][j] << "\t";
    cout << "\n";
  }  

  return cout;
}



// Quad4Prob functions

Quad4Prob::Quad4Prob() {

  // initializations
  totalcount = 0;
  for (int i=0; i < 16; i++) {
    for (int j=0; j < 4; j++) {
      probs[i][j] = 0.0;      
      counts[i][j] = 0.0;      
    }    
  }
}


void Quad4Prob::addToCount(Nucleotide before2, Nucleotide before1,
			  Nucleotide i) {
  
  // increase the total count
  totalcount++;
  
  // increase an individual count provided it's a known base
  if (i == BASE_UNKNOWN)  
    i = BASE_C;    
  if (before1 == BASE_UNKNOWN)
    before1 = BASE_C;
  if (before2 == BASE_UNKNOWN)
    before2 = BASE_C;
  
  counts[before2*4 + before1][i]++;
}


void Quad4Prob::computeProb() {
  
  // we cannot allow probability = 0....
  // first, find out how many zeros there are
  int zeros = 0;
  
  for (int i=0; i < 16; i++) 
    for (int j=0; j < 4; j++) 
      if (counts[i][j] == 0)
	zeros++;
  
  // skip if there are no zeros
  if (zeros > 0) {

    if (zeros == 64)
      cerr << "All zeros!" << endl;

    // otherwise, adjust probabilities accordingly
    double adjustment = (double)zeros / (double)(64 - zeros);
    
    for (int i=0; i < 16; i++) {
      for (int j=0; j < 4; j++) { 
	if (counts[i][j] == 0)
	  counts[i][j] = 1;
	else 
	  counts[i][j] -= adjustment;
      }
    }
  }
  
  // now, for some basic math....
  for (int i=0; i < 16; i++) 
    for (int j=0; j < 4; j++) 
      probs[i][j] = counts[i][j] / totalcount;
}


double Quad4Prob::getProb(Nucleotide before2, Nucleotide before1, 
			  Nucleotide i) {
  if (i == BASE_UNKNOWN)  
    i = BASE_C;    
  if (before1 == BASE_UNKNOWN)
    before1 = BASE_C;
  if (before2 == BASE_UNKNOWN)
    before2 = BASE_C;

  return probs[before2*4 + before1][i];
}


double Quad4Prob::getCount(Nucleotide before2, Nucleotide before1, 
			  Nucleotide i) {
  if (i == BASE_UNKNOWN)  
    i = BASE_C;    
  if (before1 == BASE_UNKNOWN)
    before1 = BASE_C;
  if (before2 == BASE_UNKNOWN)
    before2 = BASE_C;
  
  return counts[before2*4 + before1][i];
}


istream& operator>>(istream& cin, Quad4Prob& q) {
  
  cin >> q.totalcount;
  for (int i=0; i < 16; i++) 
    for (int j=0; j < 4; j++) 
      cin >> q.counts[i][j];
  for (int i=0; i < 16; i++) 
    for (int j=0; j < 4; j++) 
      cin >> q.probs[i][j];
  
  return cin;
}


ostream& operator<<(ostream& cout, Quad4Prob& q) {
  
  cout << q.totalcount << "\t";
  for (int i=0; i < 16; i++) {
    for (int j=0; j < 4; j++) 
      cout << q.counts[i][j] << "\t";
    cout << "\n";
  }
  for (int i=0; i < 16; i++) {
    for (int j=0; j < 4; j++) 
      cout << q.probs[i][j] << "\t";
    cout << "\n";
  }  

  return cout;
}
