#include "modules.h"
  
Modules::Modules(FilterSequence* seq, Registry* reg) 
{
  m_seq = seq;

  int i=0;
  seql    = seq->get_length();
  
  validPositions = new int[seql+1]; arrayInit(1, validPositions, seql+1);
  cumulativeInvalidPositions = NULL;
  
  codingl = 0;
  parseOffset = mouseParseOffset = 0;
  
  for (i=1; i<=m_seq->get_region_num(); ++i)
    if (m_seq->get_region(i)->type == REGION_CEXON) codingl += m_seq->get_region(i)->length();
  
  startCount = stopCount = -1;
  stopCodonCount     = 0;
  atgCodonCount      = 0;
  
  aligningBegins = aligningEnds = 0;
  alignRegcnt = 0;
  
  DNAToProtein0 = DNAToProtein1 = DNAToProtein2 = NULL;
  
  vector<String> repeatType;
  vector<String> htmlFile;
  
  theTupleTable      = 0;
  matrixFrame        = 0;
  matrixScore        = 0;
  matrixFProb        = 0;
  
  myData             = new double[100];    arrayZero(myData,100);

  starts             = new int[seql+1];    arrayZero(starts,             seql+1);
  stops              = new int[seql+1];    arrayZero(stops,              seql+1);
  startScores        = new double[seql+1]; arrayInit(-Infinity, startScores, seql+1);
  stopScores         = new double[seql+1]; arrayInit(-Infinity, stopScores,  seql+1);
  atgScores          = new double[seql+1]; arrayInit(-Infinity, atgScores,   seql+1);
  stopMatrix         = new SparseMatrix<int>(seql+1,seql+1,7);

  endpointType       = new int[seql+1];    arrayZero(endpointType,       seql+1);
  endpoints          = new int[seql+1];    arrayZero(endpoints,          seql+1);
  endpointStartScore = new double[seql+1]; arrayZero(endpointStartScore, seql+1);
  endpointStopScore  = new double[seql+1]; arrayZero(endpointStopScore,  seql+1);
  endpointATGScore   = new double[seql+1]; arrayZero(endpointATGScore,   seql+1);

  intexProb          = new double[seql+1];

  endptrsInv         = new int[seql+1];
  endptrType         = NULL;
  endptrs            = NULL;
  startCodon         = NULL;
  stopCodon          = NULL;
  ATGcodon           = NULL;
  atgCodons          = new int[seql+1]; arrayZero(atgCodons,  seql+1);
  stopCodons         = new int[seql+1]; arrayZero(stopCodons, seql+1);

  if (DICTIONARY_MODE) {
    proteinHits                   = new (long unsigned int)[3*(seql+1)];
    proteinSeglenHits             = new int[3*(seql+1)];
    proteinCumulativeLogs         = new double[3*(seql+1)];
    proteinCumulativeHits              = new int[3*(seql+1)];
    proteinCumulativeSeglenHits        = new int[3*(seql+1)];
    proteinCumulativeHitPositions      = new int[3*(seql+1)];
    cDNAHits                           = new (long unsigned int)[seql+1];
    proteinMaskedIntronsCumulativeHits = new int[3*(seql+1)];
    arrayZero(proteinHits,                        3*(seql+1));
    arrayZero(proteinCumulativeLogs,              3*(seql+1));
    arrayZero(proteinCumulativeHits,              3*(seql+1));
    arrayZero(proteinSeglenHits,                  3*(seql+1));
    arrayZero(proteinCumulativeSeglenHits,        3*(seql+1));
    arrayZero(proteinCumulativeHitPositions,      3*(seql+1));
    arrayZero(cDNAHits,                           seql+1    );
    arrayZero(proteinMaskedIntronsCumulativeHits, 3*(seql+1));

    proteinAccessions = new int[300000];          arrayZero(proteinAccessions,300000);
  }
  else {
    proteinHits = 0; proteinSeglenHits = 0; cDNAHits = 0;
    proteinCumulativeLogs = 0; proteinCumulativeHits = 0; proteinCumulativeSeglenHits = 0;
    proteinCumulativeHitPositions = proteinMaskedIntronsCumulativeHits = 0;
    proteinAccessions = 0;
  }

  ltt = NULL;
  intronSignals      = new int[seql+1];    arrayZero(intronSignals,   seql+1);
  alignGaps          = new int[seql+1];    arrayZero(alignGaps,       seql+1);
  alignMatches       = new int[seql+1];    arrayZero(alignMatches,    seql+1);
  alignMismatches    = new int[seql+1];    arrayZero(alignMismatches, seql+1);
  alignmentScores    = new double[seql+1]; arrayZero(alignmentScores, seql+1);

  String locus=seq->get_locus();
  
  if (DICTIONARY_MODE) { // Repeat Masking for dictionary mode
    vector<int> repeatBegin, repeatEnd, repeatRegion, 
      repeatIntronScore, repeatExonScore;
    vector<String> repeatType;  
    RepIdentify rep;
    rep.rep_find(m_seq,repeatBegin,repeatEnd,repeatType,
		 repeatRegion,repeatIntronScore,repeatExonScore,0);
    for (int i=0; i<repeatBegin.size(); ++i) { 
      if (!repeatIntronScore[i]) continue;
      for (int j = MAX(1, repeatBegin[i]); j <= MIN(repeatEnd[i], seql); ++j) validPositions[j] = 0;
    }
  }
 
  //  htmlFile.push_back("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">\n<html>\n<head>\n<title>\nMIT LCS Gene Recognition\n</title>\n</head>\n\n<body bgcolor=\"#ffffef\" text=\"#000000\" link=\"#990000\" alink=\"#ffffff\" vlink=\"#222222\">\n\n<center>\n<a href=\"http://theory.lcs.mit.edu/groups/biology.html\"><img src=\"http://plover.lcs.mit.edu/genes/images/splash.gif\"></a>\n<h1>MIT LCS Gene Recognition</h1>\n</center>\n\n<b>LOCUS: </b> <a href=\"http://www.ncbi.nlm.nih.gov/htbin-post/Entrez/query?db=n&form=6&uid="+locus+"&dopt=g\">"+locus+"</a>\n<br>\n<pre>\n</pre>");
  
  fixUnknownNuc(seq); // Remove to do long mouse!

  //  computeDictionaryInfo(reg); // Remove to do long mouse!

  humImg = mouseImg = humSeqInt = mouseSeqInt = 0;
  repeatBeginh = repeatEndh = repeatBeginm = repeatEndm = repeatIntronScoreh = repeatIntronScorem =
    repeatExonScoreh = repeatExonScorem = repeatRegionh = repeatRegionm = 0;
  
  repeatTypeh = repeatTypem = 0;

  //  loadStartStop();
  //  computeStopMatrix(reg);
  
  wfscores = NULL;
  wframes  = NULL;
}

Modules::~Modules() {
  if (myData) delete myData;
  
  if (validPositions)             delete[] validPositions;
  if (cumulativeInvalidPositions) delete[] cumulativeInvalidPositions;
  
  if (theTupleTable) {
    if (theTupleTable->freqs0) { delete[] theTupleTable->freqs0; theTupleTable->freqs0 = NULL; }
    if (theTupleTable->freqs1) { delete[] theTupleTable->freqs1; theTupleTable->freqs1 = NULL; }
    if (theTupleTable->freqs2) { delete[] theTupleTable->freqs2; theTupleTable->freqs2 = NULL; }
    if (theTupleTable->freqsI) { delete[] theTupleTable->freqsI; theTupleTable->freqsI = NULL; }
    if (theTupleTable->ranks0) { delete[] theTupleTable->ranks0; theTupleTable->ranks0 = NULL; }
    if (theTupleTable->ranks1) { delete[] theTupleTable->ranks1; theTupleTable->ranks1 = NULL; }
    if (theTupleTable->ranks2) { delete[] theTupleTable->ranks2; theTupleTable->ranks2 = NULL; }
    if (theTupleTable->ranksI) { delete[] theTupleTable->ranksI; theTupleTable->ranksI = NULL; }
    if (theTupleTable->skips)  { delete[] theTupleTable->skips;  theTupleTable->skips  = NULL; }
  }
  if (matrixFrame)   delete matrixFrame;
  if (matrixScore)   delete matrixScore;
  if (matrixFProb)   delete matrixFProb;
  
  if (wfscores) { delete[] wfscores; wfscores = NULL; }
  if (wframes)  { delete[] wframes;  wframes  = NULL; }
  
  if (starts)            delete[] starts;
  if (startScores)       delete[] startScores;
  if (stops)             delete[] stops;      
  if (stopScores)        delete[] stopScores;
  if (atgScores)         delete[] atgScores;
  if (stopCodons)        delete[] stopCodons;
  if (atgCodons)         delete[] atgCodons;
  
  /*
  if (startCodon) { delete startCodon; startCodon = NULL; }
  if (stopCodon)  { delete stopCodon;  stopCodon  = NULL; }
  if (ATGcodon)    { delete ATGcodon;    ATGcodon    = NULL; }
  */

  if (endpointType)                             delete[] endpointType;
  if (endpoints)                                delete[] endpoints;
  if (endptrsInv)                               delete[] endptrsInv;
  if (endpointStartScore)                       delete[] endpointStartScore;
  if (endpointStopScore)                        delete[] endpointStopScore;
  if (endpointATGScore)                         delete[] endpointATGScore;
  if (proteinCumulativeLogs)                    delete[] proteinCumulativeLogs;  
  if (proteinHits)                              delete[] proteinHits;
  if (proteinSeglenHits)                        delete[] proteinSeglenHits;
  if (proteinCumulativeHits)                    delete[] proteinCumulativeHits;
  if (proteinCumulativeHitPositions)            delete[] proteinCumulativeHitPositions;
  if (proteinCumulativeSeglenHits)              delete[] proteinCumulativeSeglenHits;
  if (proteinMaskedIntronsCumulativeHits)       delete[] proteinMaskedIntronsCumulativeHits;
  if (cDNAHits)                                 delete[] cDNAHits;
  if (intronSignals)                            delete[] intronSignals;
  if (alignGaps)                                delete[] alignGaps;
  if (alignMatches)                             delete[] alignMatches;
  if (alignMismatches)                          delete[] alignMismatches;
  if (alignmentScores)                          delete[] alignmentScores;
  
  if (DNAToProtein0) { delete[] DNAToProtein0; DNAToProtein0 = NULL; }
  if (DNAToProtein1) { delete[] DNAToProtein1; DNAToProtein1 = NULL; }
  if (DNAToProtein2) { delete[] DNAToProtein2; DNAToProtein2 = NULL; }

  if (proteinAccessions) { delete[] proteinAccessions; proteinAccessions = NULL; }

  if (humImg)            { delete[] humImg;            humImg            = NULL; }
  if (mouseImg)          { delete[] mouseImg;          mouseImg          = NULL; }
  if (humSeqInt)         { delete[] humSeqInt;         humSeqInt         = NULL; }
  if (mouseSeqInt)       { delete[] mouseSeqInt;       mouseSeqInt       = NULL; }
  
  if (endptrType)        { delete[] endptrType;        endptrType        = NULL; }
  if (endptrs)           { delete[] endptrs;           endptrs           = NULL; }
  
  if (repeatBeginh)      { delete[] repeatBeginh;      repeatBeginh      = NULL; }
  if (repeatEndh)        { delete[] repeatEndh;        repeatEndh        = NULL; }
  if (repeatTypeh)       { delete[] repeatTypeh;       repeatTypeh       = NULL; }
  if (repeatRegionh)     { delete[] repeatRegionh;     repeatRegionh     = NULL; }
  if (repeatIntronScoreh){ delete[] repeatBeginh;      repeatIntronScoreh= NULL; }
  if (repeatExonScoreh)  { delete[] repeatExonScoreh;  repeatExonScoreh  = NULL; }
  
  if (repeatBeginm)      { delete[] repeatBeginm;      repeatBeginm      = NULL; }
  if (repeatEndm)        { delete[] repeatEndm;        repeatEndm        = NULL; }
  if (repeatTypem)       { delete[] repeatTypem;       repeatTypem       = NULL; }
  if (repeatRegionm)     { delete[] repeatRegionm;     repeatRegionm     = NULL; }
  if (repeatIntronScorem){ delete[] repeatBeginm;      repeatIntronScorem= NULL; }
  if (repeatExonScorem)  { delete[] repeatExonScorem;  repeatExonScorem  = NULL; }
  
  if (aligningBegins)    { delete[] aligningBegins;    aligningBegins    = NULL; }
  if (aligningEnds)      { delete[] aligningEnds;      aligningEnds      = NULL; }
  if (stopMatrix) { delete stopMatrix; }
  if (intexProb) { delete[] intexProb; intexProb = NULL; }
}

void Modules::outputValidPositions(ofstream &fout) {
  for (int i=1; i<=seql; ++i)
    fout << i << ": " << validPositions[i] << "\t" << cumulativeInvalidPositions[i] << endl;
}    

double Modules::aveIntronExonProb(int start, int stop) {
  return ((intexProb[stop]-intexProb[start])/(stop-start+1));
}

void Modules::cumulativeAlignScore(double match, double mismatch, double gap){
  int i=0;
  int isgap=0;
  int *malignGaps = new int[mouseSeql+1];
  int *halignGaps = new int[seql+1];
  arrayZero(malignGaps, mouseSeql+1);
  arrayZero(halignGaps, seql+1);
  for (i=1; i<=mouseSeq->get_length(); ++i)
    malignGaps[i]=malignGaps[i-1]+(mouseImg[i]==-1);
  for (i=1; i<=seql; ++i)
    halignGaps[i]=halignGaps[i-1]+(humImg[i]==-1);
  for (i=1; i<=seql; ++i){
    isgap=(humImg[i]==-1);
    if (isgap){
      alignGaps[i]=alignGaps[i-1]+isgap;
      alignMatches[i]=alignMatches[i-1];
      alignMismatches[i]=alignMismatches[i-1];
    }
    else {
      alignGaps[i]       = halignGaps[i-1]+malignGaps[humImg[i]];
      alignMatches[i]    = alignMatches[i-1]+(m_seq->get(i)==mouseSeq->get(humImg[i]));
      alignMismatches[i] = alignMismatches[i-1]+(m_seq->get(i)!=mouseSeq->get(humImg[i]));
    }
  }
  delete[] malignGaps;
  delete[] halignGaps;
  for (i=1; i<=seql; ++i)
    alignmentScores[i]=alignGaps[i]*gap+alignMatches[i]*match
      +alignMismatches[i]*mismatch;
}



int Modules::computeIntronSignals(LongTupleTable *longTable) {
  assert(!ltt);
  ltt = longTable;

  int totalSignals=0,i=0;
  for (i=1; i<seql-ltt->tuple_length; ++i)
    if (ltt->isHit(i,*m_seq)) intronSignals[i] = ++totalSignals;
    else                      intronSignals[i] =   totalSignals;
  
  for (; i<=seql; ++i) intronSignals[i] = totalSignals;
  
  if (VERBOSE) cout << "  found a total of " << totalSignals << " intron signals" << endl;
  return totalSignals;
}

int Modules::intronSignalHits(int start, int stop, int window) {
  int from = max(1, start-window);
  int to   = min(stop+window-11, (int)seql - 11);
  if (from >= to) return 0;
  return intronSignals[to] - intronSignals[from];
}

double Modules::computeIntronExonProb() {
  double curravg=0;
  double intextotal=0;
  TupleTable *tt = getTupleTable();
  for (int pos = 1; pos<seql; ++pos) {
    if (pos>tt->tuple_length && pos<seql-tt->tuple_length){
      double intensity=(MIN(abs((log((tt->tupleProb(pos,*m_seq,0)))-
				 log(tt->tupleProb(pos,*m_seq,1)))/log(5)),1.0));
      if ((tt->tupleProb(pos,*m_seq,0)==0)&&(tt->tupleProb(pos,*m_seq,1)==0))
	intensity=0;
      if (tt->tupleProb(pos,*m_seq,1)>tt->tupleProb(pos,*m_seq,0)) 
	curravg = intensity;
      else curravg = -intensity;
    }
    else curravg=0;
    intextotal    += curravg;   intexProb[pos]=intextotal;
  }
  return intextotal;
}

double Modules::stopScore(Nucleotide x, Nucleotide y, int human) {
  assert(x == BASE_A || x == BASE_G);
  
  if (x == BASE_G) {
    assert(y == BASE_A);
    return (human) ? .3853 : .4055;
  }
  else if (y == A) return (human) ? -.1393 : -.2107;
  else             return (human) ? -.4155 : -.3711;
}


void Modules::computeEndPtrs(Registry* reg) {
  
  if (reg->lookupVal("errorMessage").length()) return;
  startStopFind(reg);
    
  assert(startCount>=0 && stopCount>=0 && atgCodonCount>=0 && stopCodonCount>=0);

  verb("Computing end pointers...");
  
  endptrType   = new int[startCount+stopCount+atgCodonCount+stopCodonCount+2];
  endptrs      = new int[startCount+stopCount+atgCodonCount+stopCodonCount+2];

  arrayZero(endpoints,    seql+1);
  arrayZero(endpointType, seql+1);
  arrayInit(-1, endptrsInv,   seql+1); // It is important to initalize to -1: means there is no endptr at pos with value -1;
  arrayZero(endptrType,   startCount + stopCount + atgCodonCount + stopCodonCount + 2);
  arrayZero(endptrs,      startCount + stopCount + atgCodonCount + stopCodonCount + 2);
  endcnt=0;

  endpoints[1] = 1; endpoints[seql] = 1;

  int i;
  for (i=0; i<startCount; ++i) {
    endpoints[starts[i]]          = endpointType[starts[i]] = START;
    endpointStartScore[starts[i]] = startScores[i];
  }
  
  for (i=0; i<stopCount; ++i) {
    endpoints[stops[i]]          = 1;
    endpointType[stops[i]]      |= STOP;
    endpointStopScore[stops[i]]  = stopScores[i];
  }

  for (i=0; i<atgCodonCount; ++i) {
    endpoints[atgCodons[i]] = 1;
    endpointType[atgCodons[i]] |= ATG;
    endpointATGScore[atgCodons[i]] = atgScores[i];
  }
  
  for (i=1; i<=seql; ++i) {
    
    if ((i>2)&&stopCodons[i-2]) {
      endpoints[i]     = 1;
      endpointType[i] |= STOP_CODON;
    }
    
    if (endpoints[i]) {
      endptrType[endcnt] = endpointType[i];
      endptrsInv[i]      = endcnt;
      assert(endcnt < startCount + stopCount + atgCodonCount + stopCodonCount + 2);
      endptrs[endcnt++]  = i;
    }
  }
  
  for (i=0; i<endcnt; ++i) assert(endptrs[i]);
}

void Modules::computeStopMatrix(Registry *reg) {
  int i,j,k;
  cumulativeInvalidPositions = new int[seql+1];
  cumulativeInvalidPositions[0] = 0;
  for (i=1; i<=seql; ++i)
    if (validPositions[i]) cumulativeInvalidPositions[i] = cumulativeInvalidPositions[i-1];
    else                   cumulativeInvalidPositions[i] = cumulativeInvalidPositions[i-1] + 1;
  computeEndPtrs(reg);
  
  int *inBetween     = new int[endcnt-1];
  int *diagonal      = new int[endcnt];
  int *internalStop  = new int[endcnt];
  arrayZero(inBetween,endcnt-1);
  arrayZero(diagonal,endcnt);
  arrayZero(internalStop,endcnt);
  int from,to,stopCode;
  stopMatrix->clear();

  verb("Computing off-diagonal...");
  for (i=0; i<endcnt-1; ++i) {                 // Computing the stopM off-diagonal;
    from=endptrs[i]; to=endptrs[i+1];
    assert(from>0 && to>0);
    
    for (j=from; (j<=to-3)&&(diagonal[i]<7); ++j) {
      if (stopCodons[j]) internalStop[i]=(diagonal[i]|=(1<<(j%3)));
    }
    if ((from<=to-2)&&(stopCodons[to-2])) diagonal[i]|=(1<<((to-2)%3));

    if (((endptrType[i]&START)||(endptrType[i]&ATG))&&
	((endptrType[i+1]&STOP)||(endptrType[i+1]&STOP_CODON))) {
      assert((0<=diagonal[i])&&(diagonal[i]<=7));
      stopCode=diagonal[i];
      if (endptrType[i+1]&STOP_CODON) {
	stopCode=internalStop[i];
	if (!(endptrType[i+1]&STOP)) 
	  stopCode|=(1<<(to%3))|(1<<((to-1)%3));
      }
      if ((endptrType[i]&ATG)&&(!(endptrType[i]&START)))
	stopCode|=(1<<((from+1)%3))|(1<<((from+2)%3));      

      assert((0<=stopCode)&&(stopCode<=7));
      // if (cumulativeInvalidPositions[to] == cumulativeInvalidPositions[from])
      stopMatrix->addEntry(from,to,stopCode);
    }
    if (i&&stopCodons[from-1]) inBetween[i-1]=(1<<((from-1)%3));
    assert((!i)||((0<=inBetween[i-1])&&(inBetween[i-1]<=7)));
  }

  for (k = 2; k<endcnt; ++k) {  // Computing all the other diagonals;    
    for (i=0; i<endcnt-k; ++i) {
      from = endptrs[i]; to = endptrs[i+k];
      internalStop[i]  = diagonal[i]|((internalStop[i+1])&7);
      diagonal[i]     |= (diagonal[i+1]&7);
      
      if (k==2) {
	if (endptrs[i+2]-endptrs[i+1] > 1) internalStop[i] |= inBetween[i];
	diagonal[i]     |= inBetween[i]; 
      }

      if (((endptrType[i]&START)  || (endptrType[i]&ATG))  &&
	  ((endptrType[i+k]&STOP) || (endptrType[i+k]&STOP_CODON))) {
	stopCode = diagonal[i];
	if (endptrType[i+k]&STOP_CODON) {
	  stopCode = internalStop[i];
	  if (!(endptrType[i+k]&STOP)) {
	    stopCode |= (1<<(to%3))|(1<<((to-1)%3));
	  }
	}
	if ((endptrType[i]&ATG)&&(!(endptrType[i]&START))) {
	  stopCode |= (1<<((from+1)%3))|(1<<((from+2)%3));
	}
	//	if (cumulativeInvalidPositions[to] == cumulativeInvalidPositions[from])
	stopMatrix->addEntry(from,to,stopCode);
      }
    }
  }  
  delete[] inBetween; 
  delete[] diagonal;
  delete[] internalStop;
  
}

TupleTable* Modules::loadTupleTable(const char* fileName, Registry* reg) {
  // Requires: valid "fileName"
  // Modifies: "this"
  // Effects: Loads the tupleTable stored in the file specified by "fileName" and
  //          returns a handle to access the tuple table. 
  
  ifstream in(fileName);
  if (VERBOSE) cout << "Loading tuple table from : " << fileName << endl;
  TupleTable *tt = new TupleTable(in);
  verb("Tuple Table loaded.");
  in.close();
  
  if (reg->lookupVal("minusOne").contains("yes",0)) 
    tt->minusSequence(*m_seq);

  matrixFrame = new SparseMatrix<int>(seql+1,seql+1,-1);
  matrixScore = new SparseMatrix<double>(seql+1,seql+1,0);
  matrixFProb = new FullMatrix<double>(seql-tt->length()+2,7);
  return (theTupleTable = tt);
}

void Modules::frameNFind(int start, int stop, 
			 Method FrameM, Method ScoreM, NormalMethod NormM, 
			 int& frame, double& score, int scoreFramesPossible, 
			 int frameFramesPossible, double factor)
  // Requires: The "FrameMatrix" and "ScoreMatrix" have already 
  //           been created, entry not already in matrix.
  //           start>0 and stop < sequence length-tuple length+1;
  // Modifies:
  // Effects: "start" is an EXON start, "stop" is an EXON stop;
{
  assert((0<=frameFramesPossible)&&(frameFramesPossible<=7));
  assert((0<=scoreFramesPossible)&&(scoreFramesPossible<=7));
  TupleTable *tt = getTupleTable();
  if (start<2) start=1; // Check this out!!!
  if (stop>getSequence()->get_length()) stop=getSequence()->get_length();
  assert(stop <= getSequence()->get_length()); // Does not hurt to assume start > 1;
  if (stop -start < tt->length()-1)
    if (VERBOSE) cout << "*** " << stop << "- " << start << "<"  << tt->length()-1 << endl;
  double fprob[7];
  // assert(stop-start >= tt->length()-1); Commented out on November 3;
  if (stop-start >= tt->length() -1) { // If the region is "long enough";
    FullMatrix<double>* fprobM=getFProb();
    
    
    for(int i=0;i<7;i++) {
      fprob[i]=fprobM->get(stop-tt->length()+1,i)-fprobM->get(MAX(1,start-1),i);
      if ((VERBOSE) && i<6 && fprob[i]<0)
	cout << start-1 << "," << stop-tt->length()+1 << ":  fprob["
	     << i << "]: " << fprob[i] << "  = "
	     << fprobM->get(stop-tt->length()+1,i)
	     << " - " << fprobM->get(start-1,i) << endl;
      //    if (i<6) assert(fprob[i]>=0);
    }
    score=frameMethod(fprob,ScoreM,scoreFramesPossible);  // Check out framesPossible
    if (NormM==Length) score=score/(pow(stop-start-tt->length()+1,factor));
    if (NormM==Absolute) score=score/fprob[6];
    frame=(int)frameMethod(fprob,FrameM,frameFramesPossible);
  }
  else {
    score = 0.000001234567890123456789; // Irrelevant number;
    if      (!(frameFramesPossible % 2)) frame = 0;
    else if (!(frameFramesPossible % 4)) frame = 1;
    else                                 frame = 2;
  }
  assert(frameFramesPossible!=7);
  assert(frame!=3);
}

int proteinMethod = 1;

void Modules::frameDictionaryFind(int start, int stop,
				  int frame, double &score, int maxMismatchInExon) {
  score = -1000;
  int zeroCount = 0;
  static double avg5logs = -14.2125019759861/(20*20*20*20*20);
  
  switch (proteinMethod){
  case 0:
    if ((zeroCount = (stop - start + 1) -
	 (proteinCumulativeHitPositions[stop    + ((frame+2)%3)*(seql+1)] -
	  proteinCumulativeHitPositions[start-1 + ((frame+2)%3)*(seql+1)])) >
	maxMismatchInExon) return;
    score = 1.0*((double)(stop-start+1)) - 2.0*((double)zeroCount);
    break;
  case 1:
    score = ( (proteinCumulativeSeglenHits[stop    + ((frame+2)%3)*(seql+1)] -
	       proteinCumulativeSeglenHits[start-1 + ((frame+2)%3)*(seql+1)])
	      - 7*(stop-start+1) );
    break;
  case 3:
    score = ( (proteinCumulativeLogs[stop    + ((frame+2)%3)*(seql+1)] -
	       proteinCumulativeLogs[start-1 + ((frame+2)%3)*(seql+1)])
	      -  avg5logs*(stop-start+1) );
  }
}

void Modules::fprobFind(Method scoreM, Registry* reg) 

  // Effects: Creates an fprob FullMatrix containing the cumulative probabilities
  //          (+=e[j], +=1/e[j]) for each of the three frames.
{
  double intronWeight=atof(reg->lookupVal("intronWeight"));

  FilterSequence *seq=getSequence();
  TupleTable     *tt=getTupleTable();

  FullMatrix<double>* fprob=getFProb();
  
  double e[3];
  int i,j;

  FProbMethod fpM;
  String s;
  if ((s=reg->lookupVal("fprobMethod")).contains("Log")) fpM=Log;
  else if (s.contains("Normal")) fpM=Normal;
  else cout << "Bad fprobMethod" << endl;

  double fp[7];
  for (j=0; j<3; j++) {
    if (fpM==Log) e[j]=tt->logFreq(*seq,1,(4-j)%3);
    else e[j] = tt->calcTupleRelFreq(*seq,1,(4-j)%3,1,intronWeight,0);

    fprob->set(1,j,fp[j]=e[j]);

    fp[j+3]=1/MAX(tt->tupleProb((4-j)%3,1,*seq),SMALL_VALUE);   

    fprob->set(1,j+3,fp[3+j]);
  }
  
  fprob->set(1,6,frameMethod(fp,scoreM,0));

  for(i=2;i<=seq->get_length()-tt->length()+1;i++) {   
    for(j=0;j<3;++j) {
      if (fpM==Log) e[j]=tt->logFreq(*seq,i,(i+3-j)%3);
      else e[j] = tt->calcTupleRelFreq(*seq,i,(i+3-j)%3,1,intronWeight,0);

      assert(e[j]<=100000);
      fprob->set(i,j,fprob->get(i-1,j)+(fp[j]=e[j]));

      fp[j+3]=1/MAX(tt->tupleProb((i+3-j)%3,i,*seq),SMALL_VALUE);
      
      fprob->set(i,j+3,fprob->get(i-1,j+3)+fp[j+3]);
    }

    fprob->set(i,6,fprob->get(i-1,6)+frameMethod(fp,scoreM,0));
  }
}

void Modules::readMethods(Registry* reg, Method& frameM, Method& scoreM, NormalMethod& normM) {
  if (reg->lookupVal("errorMessage").length()) return;

  String s=reg->lookupVal("frameMethod");

  if (s.contains("MinIndex",0))        frameM = MinIndex;
  else if (s.contains("MaxIndex",0))   frameM = MaxIndex;
  else if (s.contains("PerkIndex",0))  frameM = PerkIndex;
  else if (s.contains("SPerkIndex",0)) frameM = SPerkIndex;
  else err("Invalid frame method.");

  if ((s=reg->lookupVal("scoreMethod")).contains("SPerk")) scoreM=SPerk;
  else if (s.contains("Max"))                              scoreM = Max;
  else if (s.contains("Min"))                              scoreM = Min;
  else if (s.contains("Perk"))                             scoreM = Perk;
  else err("Invalid score method.");

  if ((s=reg->lookupVal("normalMethod")).contains("Length")) normM = Length;
  else if (s.contains("Absolute"))                           normM = Absolute;
  else if (s.contains("None"))                               normM = None;
  else err("Invalid normalization method.");
}

void Modules::frameScoreMatrixFind(Registry* reg) {
  if (reg->lookupVal("errorMessage").length()) return;

  Method frameM, scoreM;
  NormalMethod normM;

  readMethods(reg,frameM,scoreM,normM);
  fprobFind(scoreM,reg);

  SparseMatrix<int>*    frameMatrix = getFrameMatrix();
  SparseMatrix<double>* scoreMatrix = getScoreMatrix();
  frameMatrix->clear();
  scoreMatrix->clear();

  double factor=atof(reg->lookupVal("normalLengthPower"));

  // Begin finding scores and framematrix values for the potential;
  // exon regions;

  int frame,start,stop;
  double score;
  for(int i=0;i<stopMatrix->sizeNonZero();i++) {
    frameNFind(start=stopMatrix->row(i),
	       (stop=stopMatrix->col(i))-((endpointType[stop]&STOP_CODON)?3:0),
	       frameM,scoreM,normM,frame,score,0,stopMatrix->get(i),
	       factor);
    frameMatrix->addEntry(start,stop,frame);
    // if (score == 0) score = SMALL_VALUE; // Crude fix because score=0 is the default; Sep2;
    
    //    cout << " NUMBER OF REPEATS:  " << repeatBegin.size() << endl;
    
    if (cumulativeInvalidPositions[stop] - cumulativeInvalidPositions[start] > 0) score = -Infinity;

    if (score == 0) score = SMALL_VALUE; //Crude fix;
    scoreMatrix->addEntry(start,stop,score);
  }     
  assert(scoreMatrix->sizeNonZero() == stopMatrix->sizeNonZero());
  // At this point the Score Matrix and Stop Matrix should have the same;
  // number of nonzero entries.;
  
  // Now we compute the other half of the the score matrix which;
  // contains information about possible intron regions;

  /* Introns taken out of the score matrix, March 13, 1999 !; */
  if (DICTIONARY_MODE) {
    for (int i=1; i<endcnt-1; ++i) {
      if (endptrType[i]&START) { // Intron;
	for (int j=0; j<i; ++j) {
	  if (!endptrType[j]&STOP)
	    continue;
	  score=SMALL_VALUE;
	  // The first IF statement excludes introns that have length less;
	  // than 70, and ensures exons appear in positions selected;
	  
	  scoreMatrix->addEntry(endptrs[i],endptrs[j],score);
	  
	}
      }
    }
  }
  
}


void Modules::computeWindowFrameScores(Registry *reg) {
  // Effects: Computes the "wfscores" and "wframes" with the "FrameWindowParams" 
  //          associated with "fwh".
  int i=0, k=0, pos=0, start=1;

  FilterSequence *seq=getSequence();
  int seql=seq->get_length();
  int marks[MAXSEQLEN];

  TupleTable *tt  = getTupleTable();
  int         ttl = tt->length();

  if (wfscores || wframes) warn("computeWindowFrameScores called on existing results.");
  else { 
    wfscores = new double[seql+1];
    wframes  = new int[seql+1];
  }

  arrayZero(wfscores,seql+1);
  arrayInit((int)FRAME_ANY,wframes,seql+1);

  FullMatrix<double> *e = new FullMatrix<double>(3,seql+1,0);
  for (i=0; i<3; ++i) e->set(i,0,-1);                        
  // e(i,0) is never used because e is 0-based;
  
  String s;
  TupleMethod tupleM;
  if ((s=reg->lookupVal("tupleMethod")).contains("Relevant")) {
    mark_sequence(*seq,ttl,marks);
    tupleM=Relevant;
  } else if (s.contains("TupleFreqs")) tupleM=TupleFreqs;
  else if (s.contains("Everywhere")) tupleM=Everywhere;
  else err("Undefined tuple method.");

  double exonWeight   = atof(reg->lookupVal("exonWeight"));
  double intronWeight = atof(reg->lookupVal("intronWeight"));
  double penalty      = atof(reg->lookupVal("penalty"));

  for(i=1;i<=seql-ttl+1;++i)
    for(k=0;k<3;++k)
      switch(tupleM) {
      case TupleFreqs: 
	e->set(k,i,MAX(SMALL_VALUE, tt->tupleProb(k,i,*seq))); 
	break;
      case Everywhere: 
	e->set(k,i,MAX(SMALL_VALUE, tt->calcTupleRelFreq(*seq,i,k,exonWeight,intronWeight,penalty))); 
	break;
      case Relevant:
	if (marks[i]==k)
	  e->set(k,i,MAX(SMALL_VALUE, tt->calcTupleRelFreq(*seq,i,k,exonWeight,intronWeight,penalty)));
	else e->set(k,i,MAX(SMALL_VALUE, tt->calcTupleRelFreq(*seq,i,k,exonWeight,intronWeight,0)));
      }
  double fp[6],fplast[6];
  for(i=0;i<6;++i) fp[i]=0;
  
  int windowLength=atoi(reg->lookupVal("windowLength"));
  windowLength=windowLength-(windowLength%(2*ttl))+1;
  for(i=0;i<=windowLength-ttl;++i)
    for (k=0;k<3;++k) {
      fplast[k]=(fp[k]+=e->get((i+k)%3,i+start));
      fplast[3+k]=(fp[3+k]+=1/e->get((i+k)%3,i+start));
    }
    
  Method frameM, scoreM;
  NormalMethod normM;

  readMethods(reg,frameM,scoreM,normM);

  for(start=2;start<=seql-windowLength+1;++start) {
    for(k=0;k<3;++k) {
      fp[k]=fplast[(k+2)%3]-e->get((k+2)%3,start-1)+
        e->get((k+windowLength-ttl)%3,start+windowLength-ttl);
      fp[3+k]=fplast[3+(k+2)%3]-1/e->get((k+2)%3,start-1)+ 
	1/e->get((k+windowLength-ttl)%3,start+windowLength-ttl);
    }
    for(k=0;k<6;++k) fplast[k]=fp[k];

    (wframes)[pos=windowLength/2+start] = (int)frameMethod(fp,frameM,0);
    (wfscores)[pos]                     =      frameMethod(fp,scoreM,0);
  }

  delete e;
}

void Modules::setStatEvaluators(StatEvaluator *startC, StatEvaluator *stopC, StatEvaluator *atgC) {
  startCodon = startC;
  stopCodon  = stopC;
  ATGcodon    = atgC;

  atgFind(); // Remove to do long mouse!
}

void Modules::loadStartStop() {

  startCodon = new StatEvaluator();
  stopCodon  = new StatEvaluator();

  ifstream acceptorTrainFile("/data2/rosetta/tables/train_acceptor_outfile");
  startCodon->initializeBurge(acceptorTrainFile,AG_S);
  
  ifstream donorTrainFile("/data2/rosetta/tables/train_donor_outfile");
  stopCodon->initializeBurge(donorTrainFile,GT_S);
}

void Modules::printResults(ostream& out, Parse *parsetot[2], int numparses, Registry *reg) 
// Effects: Outputs the frame window results 
{
  cout << "Print results" << endl;
  Registry *header=new Registry();
  int i=0,j=0,k=0;
  long int maxProteinHits=0;
  long int maxCDNAHits=0;

  double maxscore;
  if (reg->lookupVal("errorMessage").length()) header->add("errorMessage",reg->lookupVal("errorMessage"));
  else {
    header->add("numberParses",toa(numparses));
    maxscore=maxFrameWindowScore(1,seql);
    header->add("maxWindowFrameScore",toa(maxscore));
    header->add("locus",m_seq->get_locus());
    for (i=1;i<3*(seql+1);++i) 
      if ((i%(seql+1))&&((int)proteinHits[i]>maxProteinHits)) 
	maxProteinHits=(int)proteinHits[i];
    if (maxProteinHits) header->add("maxProteinHits",toa(maxProteinHits));

    for (i=1;i<=seql;++i) 
      if ((int)cDNAHits[i]>maxCDNAHits) maxCDNAHits=(int)cDNAHits[i];
    if (maxCDNAHits) header->add("maxCDNAHits",toa(maxCDNAHits));
    cout << "Number Parses " << numparses << endl;

  }
  out << (*header);
  delete header;
  if (reg->lookupVal("errorMessage").length()) return;

  TupleTable       *tt = getTupleTable();

  int *marks = new int[seql+1];
  mark_sequence(*m_seq,tt->length(),marks);
  
  if (m_seq->get_region_num() <= 0) {
    out << "BAD SEQUENCE: NO REGIONS" << endl;
    return;
  }
  int frtransitions=0,badtransitions=0;

  SparseMatrix<int>* frameM=getFrameMatrix();
  SparseMatrix<double>* scoreM=getScoreMatrix();
  tt=getTupleTable();

  out << (char)255 << endl;
   
  if (VERBOSE) {
    cout << "PARSE SIZE is: " << parsetot[0]->size() << endl;
    for (k=0;k<(int)parsetot[0]->size();k++) {
      cout << "Parse Region #" << k << " is from " << frameM->row((*parsetot[0])[k])
	   << " to " << frameM->col((*parsetot[0])[k]) << " ; parse[" << k << "] points to "
  	   << (*parsetot[0])[k] << endl;
    }
  }
  
  int lastIntronSignalDisplayed=-12;

  for (i=1;i<=seql;++i) {
    int regiontype = 3;
    for (j=1; j<=m_seq->get_region_num(); ++j) {
      Region *r = m_seq->get_region(j);
      if (r->start <= i && r->stop >= i) {
	switch(r->type) {
	case REGION_INTRON:  regiontype = 0; break;
	case REGION_CEXON:   regiontype = 1; break;
	case REGION_NCEXON:  regiontype = 2; break;
	default:             regiontype = 3; break;
	}
      }
    }

    int guess[numparses];
    
    for (int kk=0; kk<numparses-1; kk++)
      guess[kk]=3;
    
    for (int pco=0;pco<numparses-1;pco++){
      
      for (k=0;k<parsetot[pco]->size();k++) {
	if ((frameM->row((*parsetot[pco])[k])<=i)  && 
	    (frameM->col((*parsetot[pco])[k])>=i))
	  guess[pco]=1;
      }
      
      for (k=1;k<parsetot[pco]->size();k++) {
	if ((frameM->row((*parsetot[pco])[k-1])>i)  && 
	    (frameM->col((*parsetot[pco])[k])<i))
	  guess[pco]=0;
      }
    }
    
    out << (int)(m_seq->get(i)) << " " << regiontype << " ";
    for (int pco=0;pco<numparses-1;pco++){
      out << guess[pco] << " ";
    }

    if (i>1)
      if ((((i-1)+wframes[i-1])%3) != ((i+wframes[i])%3)) {
	frtransitions++;
	if (marks[i-1] < 3 && marks[i] < 3)
	  badtransitions++;
      }

     int displayIntExAve = reg->lookupVal("displayIntExAve").contains("yes");
     if (displayIntExAve){
  if ((15<i)&&(i<(seql-15))) {
    out << aveIntronExonProb(i-15,i+15) << " ";
  }
  else {
    out << 0 << " ";
  }
     }
     else{
       out << (wfscores[i]/(1.001*maxscore)) << " ";
     }
     if (!maxProteinHits) out << "0 0 0 ";
     else {
       out << ( ( proteinMaskedIntronsCumulativeHits[i] - proteinMaskedIntronsCumulativeHits[i-1] > 0 ) ?
		(((double)proteinHits[i])/((double)maxProteinHits)) : 0 ) << " ";
       out << ( ( proteinMaskedIntronsCumulativeHits[i+(seql+1)] - proteinMaskedIntronsCumulativeHits[i-1 + (seql-1)] > 0 ) ?
		(((double)proteinHits[i+(seql+1)])/((double)maxProteinHits)) : 0 ) << " ";
       out << ( ( proteinMaskedIntronsCumulativeHits[i + 2*(seql+1)] - 
		  proteinMaskedIntronsCumulativeHits[i-1 + 2*(seql+1)] > 0 ) ?
		(((double)proteinHits[i+2*(seql+1)])/((double)maxProteinHits)) : 0 ) << " ";
     }
     if ((double)maxCDNAHits==0) out << "0 ";
     else out << (((double)cDNAHits[i])/((double)maxCDNAHits)) << " ";
     out << endpointStartScore[i] << " " << endpointStopScore[i] << " ";
     
     int displayIntronSignals = reg->lookupVal("displayIntronSignals").contains("yes");
     
     
     if ((1<=i)&&(i<=(seql-tt->length()))) {
       out << tt->tupleProb(0,i,*m_seq) << " ";
       out << tt->tupleProb(1,i,*m_seq) << " ";
      out << tt->tupleProb(2,i,*m_seq) << " ";
      if (displayIntronSignals) {
	if (intronSignals[i]>intronSignals[i-1]) {
	  out << ((255)<<16) << endl;
	  lastIntronSignalDisplayed = i;
	}
	else
	  if (lastIntronSignalDisplayed > i-12)
	    out << ((160)<<16) << endl;
	  else out << ((0)<<8) << endl;
      }
      else {
	double intensity=(MIN(abs((log((tt->tupleProb(i,*m_seq,0)))-
				   log(tt->tupleProb(i,*m_seq,1)))/log(5)),1.0)*255);
	if ((tt->tupleProb(i,*m_seq,0)==0)&&(tt->tupleProb(i,*m_seq,1)==0))
	  intensity=0;
	if (tt->tupleProb(i,*m_seq,1)>tt->tupleProb(i,*m_seq,0)) 
	  out << (((int)intensity)<<8) << endl;
	else out << (((int)intensity)<<16) << endl;
      } 
    } else out << "0 0 0 0" << endl;
    
  }

  out << (char)255 << endl;

  if (frameM->sizeNonZero()) {
    for(i=0;i<frameM->sizeNonZero();i++) {
      out << frameM->row(i) << " " << frameM->col(i) << " ";
      out << scoreM->get(i) << " " << (1<<frameM->get(i)) << " ";
      out << (1<<((frameM->row(i)-frameM->get(i)+3)%3));
      out << " " << (1<<((frameM->col(i)+4-frameM->get(i))%3));
      out << " " << stopMatrix->get(i) << endl;
    } 
  } else out << endl;

  out << (char)255 << endl;

  for(i=0;i<repeatNumberh;i++) {
    out << repeatBeginh[i] << " " << repeatEndh[i];
    out << " " << repeatTypeh[i] << endl;
  }

  out << (char)255 << endl;

  ofstream outFile("/cpp/java/Genes/sequence.html");
  for(i=0;i<htmlFile.size();i++) outFile << htmlFile[i];
  outFile << endl;
  outFile << "<p><hr><p><h2>Reference</h2>\n<p><ul><li><a href=\"http://psyche.uthct.edu/shaun/SBlack/geneticd.html\">The genetic code</a></li>\n<li><a href=\"http://linkage.rockefeller.edu/wli/gene/list.html\">Bibliography on gene recognition</a></li></ul>\n<h2>Databases</h2><ul><li><a href=\"http://bmbsgi11.leeds.ac.uk/bmb5dp/owl.html\">Owl</a></li>\n<li><a href=\"http://www3.ncbi.nlm.nih.gov/Entrez/\">Enterez</a></li>\n<li><a href=\"http://www.ncbi.nlm.nih.gov/dbEST/index.html\">dbEST</a></li></ul>\n<h2>Tools</h2><ul><li><a href=\"http://www.ncbi.nlm.nih.gov/BLAST/\">BLAST</a></li></ul>" << endl;
  outFile.close();

  out << "http://plover.lcs.mit.edu/genes/sequence.html" << endl;

  cout << "Done!" << endl;
}
 


void Modules::atgFind() {

  int i;
  atgCodonCount = 0;

  for(i=1;i<=seql-3;++i) {
    if ((stopCodons[i]=isStopCodon(m_seq->get(i),m_seq->get(i+1),m_seq->get(i+2)))) stopCodonCount++;

    if ((m_seq->get(i)==BASE_A)&&(m_seq->get(i+1)==BASE_T)&& (m_seq->get(i+2)==BASE_G)) {
      if (i > 5 && i < seql-6) {
	double score1, score2;
	
	ATGcodon->burgeEvalATG(m_seq, i, score1, score2);
	atgScores[atgCodonCount] = score2;
	atgCodons[atgCodonCount] = i;

	if (atgCodonCount) atgScores[atgCodonCount] += MIN(0.0,atgScores[atgCodonCount] - atgScores[atgCodonCount-1]); // Left Rule;
	atgCodonCount++;
      }
      else {
	atgScores[atgCodonCount] = 0;
	atgCodons[atgCodonCount] = i;
	atgCodonCount++;
      }
    }
  }
}

void Modules::startStopFind(Registry* reg) 
// Effects: Finds start and stops of introns.
{
  cout << "Finding Splice Sites" << endl;

  FilterSequence* seq = getSequence();
  startCount = stopCount = 0;

  String s;

  int pos;

  double *newStartScores = new double[seql+1];
  double *newStopScores  = new double[seql+1];

  startCodon->evaluateStarts(seq, newStartScores);
  stopCodon->evaluateStops(  seq, newStopScores);

  int startsAbove0 = 0, stopsAbove0 = 0;

  for(pos=3; pos <= seql-2; pos++) {
    if (!validPositions[pos]) continue;

    if (newStartScores[pos] > -Infinity) {
      if (!DICTIONARY_MODE || newStartScores[pos] >= -7) {
	startScores[startCount] = newStartScores[pos];
	starts[startCount]      = pos;
	if (startScores[startCount]>0) startsAbove0++;
	++startCount;
      }
    }
    if (newStopScores[pos] > -Infinity) {
      if (!DICTIONARY_MODE || newStopScores[pos] >= -2) {
	stopScores[stopCount] = newStopScores[pos];
	stops[stopCount]      = pos;
	if (stopScores[stopCount]>0) stopsAbove0++;
	++stopCount;
      }
    }
  }
  cout << "Start Count: " << startCount << "   >0: " << startsAbove0 << endl;
  cout << "Stop Count: " << stopCount << "   >0: " << stopsAbove0 << endl;
  delete[] newStartScores;
  delete[] newStopScores;
}

int Modules::framesPossible(int regionA, int regionB) {
  // Calculate which frames are possible when regionA and regionB are concatenated...

  if (regionA==-1) return stopMatrix->get(regionB);
  if (regionB==-1) return stopMatrix->get(regionA);

  int stopCode=stopMatrix->get(regionA);
  int regBcode=stopMatrix->get(regionB);

  int endA=stopMatrix->col(regionA);
  int startB=stopMatrix->row(regionB);

  // Take care of frame shift due to intervening intron.
  stopCode|=(((regBcode<<3)|regBcode)>>((startB-endA-1)%3))&7; 

  // Take care of stop codon positions that are created by concatenating the two exons.
  if (isStopCodon(m_seq->get(endA-1),m_seq->get(endA),m_seq->get(startB))) 
    stopCode|=(1<<((endA-1)%3));
  if (isStopCodon(m_seq->get(endA),m_seq->get(startB),m_seq->get(startB+1)))
    stopCode|=(1<<(endA%3));

  return stopCode;
}

double Modules::maxFrameWindowScore(int start, int stop) {
  assert((start>0)&&(start<=seql));
  assert((stop>0)&&(stop<=seql));
  assert(stop>=start);
  double maxscore=0;
  for (int i=start;i<=stop;++i)
    if (wfscores[i]>maxscore) maxscore = wfscores[i];
  return maxscore;
}

Parse* Modules::newGenerateParse(Registry* reg) {
  int useProteinHomology = reg->lookupVal("generateParse").contains("protein");
  int usecDNAHomology = reg->lookupVal("generateParse").contains("cDNA");

  if (!((reg->lookupVal("generateParse")).contains("frametest") ||
	useProteinHomology || usecDNAHomology))
    return new Parse;
  
  
  SparseMatrix<double>*   scoreMatrix = getScoreMatrix();
  SparseMatrix<int>*      frameMatrix = getFrameMatrix();
  HashSparseMatrix<int>*  nonZeroMap  = new HashSparseMatrix<int>(endcnt,endcnt,-1);

  int i,j,k;
  for(i=0;i<scoreMatrix->sizeNonZero();i++) {
    int row_i = scoreMatrix->row(i), col_i = scoreMatrix->col(i);
    assert(endptrsInv[row_i] != -1 && endptrsInv[col_i] != -1);
    nonZeroMap->addEntry(endptrsInv[row_i],endptrsInv[col_i],i);
  }
  cout << "Generating parse..." << endl;
  double frameTestWeightIntron  = atof(reg->lookupVal("frameTestWeightIntron"));
  int    frameTestLog           = reg->lookupVal("frameTestLog").contains("yes");
  Method frameM, scoreM;
  NormalMethod normM;

  readMethods(reg,frameM,scoreM,normM);

  double spliceScoresWeight     = atof(reg->lookupVal("spliceScoresWeight"));
  double avgIndicatingExon      = atof(reg->lookupVal("avgIndicatingExon"));
  double factor                 = atof(reg->lookupVal("normalLengthPower"));
  double differentFramePenalty  = atof(reg->lookupVal("differentFramePenalty"));
  int    minimumIntronLength    = atoi(reg->lookupVal("minimumIntronLength"));  
  int    maxMismatchInExon      = atoi(reg->lookupVal("maxMismatchInExon"));

  int minFrametestExonLength    = atoi(reg->lookupVal("minFrametestExonLength"));
  int intronHitsCutoff          = atoi(reg->lookupVal("intronHitsCutoff"));

  double      ***sites = new (double**)[endcnt];
  int     ****siteBptr = new (int***)[endcnt];
  for (i=0; i<endcnt; ++i) {
    sites[i]    = new (double*[6]);
    siteBptr[i] = new (int**[6]);
    for (j=0; j<6; ++j) {
      siteBptr[i][j] = new int*[3];
      sites[i][j]    = new double[3];
      for (k=0; k<3; ++k) {
	sites[i][j][k]                = -Infinity;
	siteBptr[i][j][k]             = new int[3];
	siteBptr[i][j][k][ENDPTR]     = -1;         // Invalid endpointer;
	siteBptr[i][j][k][ENDPTRTYPE] = -1;         // Invalid endpointer type;
	siteBptr[i][j][k][FRAME]      = -1;         // Invalid frame;
      }
    }
  }

  for (k=0; k<3; ++k) sites[0][BEGINNING][k] = 0;

  int fr,lastfr,frame,maxframe,maxj,currType,lastType;
  double maxscore, score, intronScore, currentScore;

  // DYNAMIC PROGRAMMING LOOP STARTS HERE;
  // Counter i keeps track of the endpoint of the current region
  // being examined. Counter j is the left endpoint. 

  int surviving[20]; arrayZero(surviving,20);

  for (i=1; i<endcnt-1; ++i) { // Main iteration - end of current region;
    if (endptrType[i]&ATG) {
      for (fr=0;fr<3;fr++) {
	sites[i][START_CODING][fr] = ( (fr == (endptrs[i]%3)) ? 0 : -Infinity );
	siteBptr[i][START_CODING][fr][ENDPTR] = 0;
	siteBptr[i][START_CODING][fr][ENDPTRTYPE] = BEGINNING;
	siteBptr[i][START_CODING][fr][FRAME] = 0;
      }
    }
    
    if (endptrType[i]&START) { // Intron;
      maxscore = -Infinity;
      for (j=0; j<i; ++j) {
	if (VERBOSE) cout << "Intron: (" << endptrs[j] << "," << endptrs[i] << ")" << endl;

	// The first IF statement excludes introns that have length less;
	// than 70, and ensures exons appear in positions selected;

	if ((!(endptrType[j]&STOP)) || (endptrs[j]+minimumIntronLength>=endptrs[i])) {
	  if (VERBOSE) cout << "\t reject 1" << endl;
	  continue;
	}
	if (nonZeroMap->get(i,j) == -1) {
	  if (VERBOSE) cout << "\t reject 2" << endl;
	  continue;
	}
	assert(endpointStartScore[endptrs[j]] > -INF && endpointStopScore[endptrs[i]] > -INF);

	score = 0;

	if (frameTestWeightIntron!=0) {       // Frametest
	  frameNFind(endptrs[j]+1,endptrs[i]-1,frameM,scoreM,normM,
		     frame,intronScore,0,0,factor);
	  if (frameTestLog) intronScore=log(MAX(SMALL_VALUE,intronScore));
	  score -= frameTestWeightIntron*intronScore;
	}
	
	for (fr=0;fr<3;fr++) {
	  if (VERBOSE) cout << "\t frame: " << fr << endl;
	  for (lastfr = 0; lastfr<3; ++lastfr) {
	    //	    cout << "\t\t lastfr: " << endl;
	    currentScore = score+sites[j][GT][lastfr];
	    if (currentScore == -Infinity && VERBOSE) cout << "\t\t\t reject 2.5" << endl;
	    if ((lastfr == (endptrs[j]-2)%3) &&
		isStopCodon(m_seq->get(endptrs[j]-2),
			    m_seq->get(endptrs[j]-1),
			    m_seq->get(endptrs[j]))) {
	      if (VERBOSE) cout << "\t\t\t reject 3" << endl;
	      currentScore = -Infinity;
	    }
	    
	    if (!(((lastfr+endptrs[i]-endptrs[j]+5)%3==fr)&& // Forces frame consistency;
		  (!(((lastfr==((endptrs[j]-1)%3))                &&
		      (isStopCodon(m_seq->get(endptrs[j]-1),
				   m_seq->get(endptrs[j]),
				   m_seq->get(endptrs[i]))))  ||
		     ((lastfr==(endptrs[j]%3))                    &&
		      isStopCodon(m_seq->get(endptrs[j]),
				  m_seq->get(endptrs[i]),
				  m_seq->get(endptrs[i]+1))))))) {
	      if (VERBOSE) cout << "\t\t\t reject 4" << endl;
	      currentScore =  -Infinity;
	    }

	    if (VERBOSE) cout << "\t\t\t current score: " << currentScore << endl;
	    if (currentScore > sites[i][AG][fr]) {  // Compares score with previous best;
	      if (VERBOSE) cout << "\t\t\t accept with score: " << currentScore << endl;
	      sites[i][AG][fr]                = currentScore;
	      siteBptr[i][AG][fr][ENDPTR]     = j;
	      siteBptr[i][AG][fr][ENDPTRTYPE] = GT;
	      siteBptr[i][AG][fr][FRAME]      = lastfr;
	    }
	  }
	}
      }
    }
    if ((endptrType[i]&STOP)||(endptrType[i]&STOP_CODON)) { // Exon;
      int region, stopCode;
      for (j=0; j<i; ++j) {

	if (VERBOSE) cout << "Exon: (" << endptrs[j] << "," << endptrs[i] << ")" << endl;
	int isInternal=0;
	if ((!(endptrType[i]&STOP)) && (endptrType[j]&START)) {
	  surviving[0]++;
	  isInternal=true;
	}
	if (endptrs[j]+2000<endptrs[i])        continue; // Exon too long;
	if (isInternal) surviving[1]++;
	if ((region=nonZeroMap->get(j,i))==-1) continue; // Invalid exon;
	if (isInternal) surviving[2]++;
	assert((endptrType[j] & ATG) || (endptrType[j] & START));
	for (int rType = 0; rType < 4; ++rType) {
	  if (!((((  rType&1) &&(endptrType[j]&ATG))||
		 ((!(rType&1))&&(endptrType[j]&START)))&&
		(((  rType&2) &&(endptrType[i]&STOP_CODON))||
		 ((!(rType&2))&&(endptrType[i]&STOP)))))       continue;
	  if (VERBOSE) cout << "\t rType: " << rType << endl;
	  stopCode=stopMatrix->get(region);
	  if (isInternal) surviving[3]++;
	  if (rType&1) stopCode|=(1<<((endptrs[j]+1)%3))|(1<<((endptrs[j]+2)%3));      
	  if (rType&2) stopCode|=(1<<(endptrs[i]%3))|(1<<((endptrs[i]-1)%3));
	  double lengthNormal = ((double) endptrs[i] - (double) endptrs[j]) / 100.0;

	  for (fr=0;fr<3;fr++) {
	    if ((1<<fr)&stopCode)                                          continue;  // Continue only if frame fr is possible;
	    if (VERBOSE) cout << "\t\t frame: " << fr << endl;
	    if (isInternal) surviving[4]++;
	    if (sites[j][ (rType&1)? START_CODING : AG ][fr] == -Infinity) {
	      if (VERBOSE) cout << "\t\t\t partial parse not possible" << endl;
	      continue;  // Continue only if current partial parse is possible;
	    }
	    if (isInternal) surviving[5]++;
	    if (VERBOSE) cout << fr;
	    if (useProteinHomology) {
	      frameDictionaryFind(endptrs[j], endptrs[i] - ((endptrType[i]&STOP_CODON) ? 3 : 0),
				  fr,score,maxMismatchInExon);
	      
	      /*
	      // New code for between-exon scoring, Jan 11, begins here;
	      if (!(rType&1)) {
		int i_last = siteBptr[j][ AG ][fr][ENDPTR]; assert(endptrType[i_last] & STOP);
		int lastfr = siteBptr[j][ AG ][fr][FRAME];  assert(lastfr >= 0 && lastfr <= 2);
		
		if (endptrs[i_last] > 31 && endptrs[i] < seql-31) {
		  char exonsText[61];
		  int offset;
		  if      (lastfr == (endptrs[i_last]-1)%3) offset = 13;
		  else if (lastfr == (endptrs[i_last]-2)%3) offset = 14;
		  else if (lastfr == (endptrs[i_last]-3)%3) offset = 15;
		  else assert(0);
		  int filled = 0;
		  for (int u = endptrs[i_last]-offset; u <= endptrs[i_last]; ++u)
		    exonsText[filled++] = m_seq->base2char(m_seq->get(u));
		  for (int u = endptrs[j]; filled < 60; ++u)
		    exonsText[filled++] = m_seq->base2char(m_seq->get(u));
		  exonsText[filled] = '\0';
		  
		  char *proteinText = m_seq->DNAToProtein(exonsText);
		  ivector<long unsigned int> proteinSegpos;
		  ivector<long unsigned int> proteinSeglen;
		  ivector<long unsigned int> proteinSegacc;
		  ivector<long unsigned int> proteinImage;

		  owl->Segments(proteinText,seql+3,MAX(8,proteinCutoff),proteinSegpos, proteinImage, proteinSeglen, proteinSegacc);
		  int segCount = proteinSegpos.size();
		  int boundaryHitPositions[20] = { 0 };
		  int boundarySeglenHits[20] = { 7 };
		  for (int segment = 0; segment < segCount; ++segment) {
		    if (proteinAccessions[proteinSegacc[segment]]) continue;
		    if (proteinSegpos[segment] >= 9 || proteinSegpos[segment] + proteinSeglen[segment] <= 10) continue;
		    switch (proteinMethod) {
		    case 0:
		      for (int u = proteinSegpos[segment]; 
			   u <= proteinSegpos[segment] + proteinSeglen[segment]; 
			   boundaryHitPositions[u++] = 1);
		      break;
		    case 1:
		      if (proteinSeglen[segment] <= 7) continue;
		      for (int u = proteinSegpos[segment];
			   u <= proteinSegpos[segment] + proteinSeglen[segment];
			   u++) boundarySeglenHits[u] = max(boundarySeglenHits[u], proteinSeglen[segment]);
		      break;
		    }
		  }
		  for (int u=0; u<20; ++u)
		    switch (proteinMethod) {
		    case 0: score += boundaryHitPositions[u];        break;
		    case 1: score += MIN(2,boundarySeglenHits[u]-7); break;
		    }
		}
	      }
	      // Jan 11 code ends here;
	      */

	      frame = fr;
	      if (isInternal) {
		if (score >= 0) surviving[7]++;
		if (score >= -Inf) surviving[6]++;
	      }
	    }
	    else if (usecDNAHomology) {
	      frameDictionaryFind(endptrs[j], endptrs[i] - ((endptrType[i]&STOP_CODON) ? 3 : 0),
				  fr,score,maxMismatchInExon);
	      frame = frameMatrix->get(region);
	    }
	    else {
	      score = scoreMatrix->get(region);
	      frame = frameMatrix->get(region);
	      
	      score -= avgIndicatingExon * (endptrs[i]-endptrs[j]+1);
	      if ((aveIntronExonProb(endptrs[j],endptrs[i]))<0){         // CUTOFF (A);
		score +=50*(aveIntronExonProb(endptrs[j],endptrs[i]));
	      }
	      if (endptrs[i]-endptrs[j]+1 < minFrametestExonLength) score = -50;
	      if (frameTestLog) score=log(MAX(SMALL_VALUE,score));
	    }

	    if (intronSignalHits(endptrs[j],endptrs[i], 0) >= 
		lengthNormal * (double)intronHitsCutoff) score = -Infinity;
	    else score -= MAX(score, -score) * .1 * intronSignalHits(endptrs[j], endptrs[i], 0); // was *.1;

	    if (fr != frame) score -= differentFramePenalty * lengthNormal*lengthNormal;

	    double SSscore;
	    if (rType&1) SSscore  = 2*endpointATGScore[endptrs[j]];
	    else         SSscore  = endpointStartScore[endptrs[j]]*1;
	    
	    if (rType&2) SSscore += 0;
	    else         SSscore += endpointStopScore[endptrs[i]]*7;
	    
	    switch (rType) {
	    case 0: if (SSscore < -5) { if (VERBOSE) cout << "\t\t\t\t bad splice sites: " << SSscore << endl; continue; } break;
	    case 1:
	    case 2: if (SSscore < -5) { if (VERBOSE) cout << "\t\t\t\t bad splice sites: " << SSscore << endl; continue; } break;
	    case 3: break;
	    }
	    if (VERBOSE) cout << "\t\t\t\t SS Score: " << SSscore << endl;
	    
	    if (score >=0) score += spliceScoresWeight * SSscore;
	    
	    if (isInternal) {
	      if (score >= -Inf) surviving[8]++;
	      if (score >= 0) surviving[9]++;
	    }
	    currType     = (rType&2) ? STOP_CODING  : GT;
	    lastType     = (rType&1) ? START_CODING : AG;
	    currentScore = score + sites[j][lastType][fr];

	    if (VERBOSE) cout << "\t\t\t score: " << currentScore << "\t prev. score: (" << endptrs[siteBptr[i][currType][fr][ENDPTR]] << "," << endptrs[i] << "): "
			      << sites[i][currType][fr] << endl;
	    if (currentScore>sites[i][currType][fr]) {
	      if (isInternal) surviving[10]++;

	      sites[i][currType][fr] = currentScore;
	      siteBptr[i][currType][fr][ENDPTR]     = j;
	      siteBptr[i][currType][fr][ENDPTRTYPE] = lastType;
	      siteBptr[i][currType][fr][FRAME]      = fr;
	    }
	  }
	}
      }
    }
  }
  cout << "surviving: ";
  for (i=0; i<20; ++i) cout << surviving[i] << " ";
  cout << endl << endl;
  
  maxscore=-Infinity;
  maxframe=0;
  maxj=0;
  for(i=0;i<endcnt;i++) 
    for(fr=0;fr<3;fr++) 
      if (sites[i][STOP_CODING][fr]>maxscore   &&
	  fr == (endptrs[i]-2)%3               &&
	  isStopCodon(m_seq->get(endptrs[i]-2),
		      m_seq->get(endptrs[i]-1),
		      m_seq->get(endptrs[i]))) {
	maxscore=sites[i][STOP_CODING][fr];
	maxj=i;
	maxframe=fr;
      }

  // Convert to special parse format
  i=maxj;
  fr=maxframe;
  currType=STOP_CODING;
  Parse* parse=new Parse;

  while (i>0) {
    lastType = siteBptr[i][currType][fr][ENDPTRTYPE];
    j        = siteBptr[i][currType][fr][ENDPTR];
    if (( (lastType == START_CODING) || (lastType == AG))  &&
	( (currType == STOP_CODING ) || (currType == GT))) {
      parse->push_back(nonZeroMap->get(j,i));

    }
    fr       = siteBptr[i][currType][fr][FRAME];
    currType = lastType;
    i        = j;
  }

  cout << "Finished parse!  Score: " << maxscore << endl;

  for (i=0; i<endcnt; ++i) {
    for (j=0; j<6; ++j) {
      for (k=0; k<3; ++k) 
	delete[] siteBptr[i][j][k];
      delete[] sites[i][j];
      delete[] siteBptr[i][j];
    }
    delete[] sites[i];
    delete[] siteBptr[i];
  }
  delete[] sites;
  delete[] siteBptr;
  delete nonZeroMap;
  return parse;
}


void Modules::setAligningRegions(ifstream &fin1) {

  int i,j;
  char dummyBuf[100];
  fin1 >> dummyBuf; fin1 >> alignRegcnt; assert(alignRegcnt>0);
  aligningBegins = new int[alignRegcnt]; arrayZero(aligningBegins, alignRegcnt);
  aligningEnds   = new int[alignRegcnt]; arrayZero(aligningEnds,   alignRegcnt);
  
  for (int j=0; j<alignRegcnt; ++j) {
    fin1 >> dummyBuf; fin1 >> aligningBegins[j]; fin1 >> aligningEnds[j]; assert(aligningEnds[j] > aligningBegins[j]);
  }
   
  for (i=0; i<alignRegcnt-1; ++i)
    for (j=aligningEnds[i]+40; j<aligningBegins[i+1]-41; ++j) validPositions[j]=0;    
}

void Modules::setRepeats(ifstream &fin) {
  
  setHumReps(fin);
  for (int i=0; i<repeatNumberh; ++i) {
    if (!repeatIntronScoreh[i]) continue;
    for (int j = MAX(1, repeatBeginh[i]); j <= MIN(repeatEndh[i], seql); ++j) validPositions[j] = 0;
  }
}

void Modules::setRepeats(ifstream &finh, ifstream &finm) {
  setRepeats(finh);
  setMouseReps(finm);
  
  for (int i=0; i<repeatNumberm; ++i) {
    if (!repeatIntronScorem[i]) continue;
    for (int j = MAX(repeatBeginm[i],1); j <= MIN(repeatEndm[i], mouseSeql); ++j)
      if (mouseImg[j] > 0 && mouseImg[j] <= seql) {
	validPositions[mouseImg[j]] = 0;
      }
  }
  int lastInvalid = -15;
  for (int i=1; i<seql; ++i)
    if (validPositions[i] && !validPositions[i+1] && lastInvalid > i-10)
      for (int j=lastInvalid+1; j<=i; ++j) validPositions[j] = 0;
    else if (!validPositions[i]) lastInvalid = i;
}
		       

void Modules::setHumReps(ifstream &fin) {
  int var, i;
  String name;
  
  assert(fin.good());
  fin >> repeatNumberh;
  repeatBeginh       = new int[repeatNumberh];      arrayInit(0,repeatBeginh, repeatNumberh);
  repeatEndh         = new int[repeatNumberh];      arrayInit(0,repeatEndh, repeatNumberh);
  repeatTypeh        = new String[repeatNumberh];     
  repeatRegionh      = new int[repeatNumberh];      arrayInit(0,repeatRegionh, repeatNumberh);
  repeatIntronScoreh = new int[repeatNumberh];      arrayInit(0,repeatIntronScoreh, repeatNumberh);
  repeatExonScoreh   = new int[repeatNumberh];      arrayInit(0,repeatExonScoreh, repeatNumberh);
  
  for (i=0; i<repeatNumberh; i++) {
    fin >> var;
    repeatBeginh[i]=var-parseOffset;
    fin >> var;
    repeatEndh[i]=var-parseOffset; 
    fin >> name;
    repeatTypeh[i]=name;
    fin >> var;
    repeatRegionh[i]=var;
    fin >> var;
    repeatIntronScoreh[i]=var;
    fin >> var;
    repeatExonScoreh[i]=var;
  }
}

void Modules::setMouseReps(ifstream &fin) {
  int var, i;
  String name;
  
  fin >> repeatNumberm;
  repeatBeginm       = new int[repeatNumberm];      arrayInit(0,repeatBeginm, repeatNumberm);
  repeatEndm         = new int[repeatNumberm];      arrayInit(0,repeatEndm, repeatNumberm);
  repeatTypem        = new String[repeatNumberm];     
  repeatRegionm      = new int[repeatNumberm];      arrayInit(0,repeatRegionm, repeatNumberm);
  repeatIntronScorem = new int[repeatNumberm];      arrayInit(0,repeatIntronScorem, repeatNumberm);
  repeatExonScorem   = new int[repeatNumberm];      arrayInit(0,repeatExonScorem, repeatNumberm);

  for (i=0; i<repeatNumberm; i++) {
    fin >> var;
    repeatBeginm[i]=var-mouseParseOffset;
    fin >> var;
    repeatEndm[i]=var-mouseParseOffset;
    fin >> name;
    repeatTypem[i]=name;
    fin >> var;
    repeatRegionm[i]=var;
    fin >> var;
    repeatIntronScorem[i]=var;
    fin >> var;
    repeatExonScorem[i]=var;
  }
}

void Modules::setImages(ifstream &fin1, ifstream &fin2) {
  setHumImg(fin1);
  setMouseImg(fin2);
}

void Modules::setHumImg(ifstream &fin) {
  humImg   = new int[seql+1];
  humImg[0] = -1;

  char dummy[100];
  fin >> dummy;
  assert(dummy[0] == 'R');
  cout << dummy << endl;

  int hbegin, hend;
  fin >> hbegin; fin >> hend;
  assert(hend-hbegin + 1 == seql);

  parseOffset = hbegin-1;

  for (int i=0; i<=seql; ++i) {
    fin >> humImg[i];
    assert(humImg[i] >= -1);
  }
}
void Modules::setMouseImg(ifstream &fin) {
  char dummy[100];
  fin >> dummy;
  assert(dummy[0] = 'R');

  int mbegin, mend; 
  fin >> mbegin; fin >> mend;
  mouseSeql = mend-mbegin+1;
  mouseParseOffset = mbegin-1;

  mouseImg   = new int[mouseSeql+1];
  mouseImg[0] = -1;

  for (int i=0; i<=mouseSeql; ++i) {
    fin >> mouseImg[i];
    assert(mouseImg[i] >= -1);
  }
}


double Modules::exonPairScore(int hstart, int hstop, int hfr, int mstart, int mstop, int mfr, int rType) {
  
  double score = -Infinity;
  double diffLengthMod3Penalty = -9;
  double diffLengthPenalty     = -3;
  double diffLengthMod3PenaltyInternal = -27;
  double diffLengthPenaltyInternal     = -9;

  register int i,j;
  int haastart, maastart, haastop, maastop;
  int haalen, maalen;
  char *haa    = new char[haalen = (hstop-hstart+3)/3 +1];
  char *maa    = new char[maalen = (mstop-mstart+3)/3 +1];
  int  *haaimg = new int[haalen];
  int  *maaimg = new int[maalen];

  int halignStart, halignStop, malignStart, malignStop;
  halignStart = (rType&1) ? hstart + 3 : hstart;
  malignStart = (rType&1) ? mstart + 3 : mstart;
  halignStop  = (rType&2) ? hstop  - 3 : hstop;
  malignStop  = (rType&2) ? mstop  - 3 : mstop;

  if (!rType) {
    assert(hstop-hstart+1 >= 15);
    assert(mstop-mstart+1 >= 9);
  }
  else assert(hstop-hstart+1>=12 && mstop-mstart+1>=9);
  
  alignExonPair(m_seq, mouseSeq, halignStart, halignStop, malignStart, malignStop, hfr, mfr, humImg, mouseImg, haa, maa, 
		haastart, maastart, haastop, maastop, haaimg, maaimg);

  assert(!((haastop-haastart+1)%3) && (haastop-haastart+1)/3 < haalen); haalen = (haastop-haastart+1)/3;
  assert(!((maastop-maastart+1)%3) && (maastop-maastart+1)/3 < maalen); maalen = (maastop-maastart+1)/3;
  
  if (rType&1) assert(haastart == hstart+3 && maastart == mstart+3);
  if (rType&2) assert(haastop  == hstop -3 && maastop  == mstop -3);

  double aaScore = 0;
  int gapCnt = 0;
  for (j=0; j<haalen; ++j)
    if (haaimg[j] >=0) aaScore += pamMat[aaOrder[aa_char_num[haa[j]]]][aaOrder[aa_char_num[maa[haaimg[j]]]]];
    else {
      gapCnt++;
      aaScore += pamMat[aaOrder[aa_char_num[haa[j]]]][21];
    }
  for (j=0; j<maalen; ++j)
    if (maaimg[j] ==-1) {
      gapCnt++;
      aaScore += pamMat[21][aaOrder[aa_char_num[maa[j]]]];
    }

  if (VERBOSE) cout << "Hum: " << haa << "\t" << endl;
  if (VERBOSE) cout << "Mus: " << maa << "\t" << endl;

  double randomCodon = 1.0 / 61.0;
  double codonScore  = 0.0;

  for (i = haastart; i<haastop; i += 3) codonScore += log(hcodonMat[  m_seq->get(i)   ][  m_seq->get(i+1)   ][  m_seq->get(i+2)   ] / randomCodon);
  for (i = maastart; i<maastop; i += 3) codonScore += log(mcodonMat[ mouseSeq->get(i) ][ mouseSeq->get(i+1) ][ mouseSeq->get(i+2) ] / randomCodon);
  
  
  if (VERBOSE)  cout << "Gap Count: " << gapCnt << "\t Current score: " 
		     << aaScore << "+" << codonScore << " = " << aaScore+codonScore;
  if (aaScore+codonScore > score) {
    score = aaScore + codonScore;
    if (VERBOSE) cout << "\t updating score";
  }
  if (VERBOSE) cout << endl;
  
  // Length scores:
  
  switch (rType) {
  case 0:
    score += 0.75 * ( MIN(0.0, ((double)haalen -  20.0) /  1.0) + MIN(0.0, ((double)maalen -  20.0) /  1.0) );
    break;
  case 1: 
  case 2:
    break;
  case 3:
    score += 0.5 * ( MAX(0.0, ((double)haalen - 333.0) / 6.0)  + MAX(0.0, ((double)maalen - 333.0) /  6.0) );
    score += 0.5 * ( MIN(0.0, ((double)haalen - 200.0) / 9.0)  + MIN(0.0, ((double)maalen - 200.0) /  9.0) );
    score += 0.5 * ( MAX(0.0, ((double)haalen - 240.0) / 9.0)  + MAX(0.0, ((double)maalen - 250.0) /  9.0) );
    break;
  }
  
  if (VERBOSE) cout << "\t score becomes: " << score;
  switch (rType) {
  case 0:
    if (hstop-hstart     != mstop-mstart)     score += diffLengthPenaltyInternal;
    if ((hstop-hstart)%3 != (mstop-mstart)%3) score += diffLengthMod3PenaltyInternal;
    break;
  case 1:
  case 2:
  case 3:
    if (hstop-hstart     != mstop-mstart)     score += diffLengthPenalty;
    if ((hstop-hstart)%3 != (mstop-mstart)%3) score += diffLengthMod3Penalty;
    break;
  }



  /*
  if (VERBOSE) cout << "\t ...: " << score;

  // Mapping of splice sites;
  
  double matchSSleft=0, matchSSright=0;
  switch (rType) {
  case 0:
    if (humImg[hstart] == mstart) for (i=-3; i<=0; ++i) matchSSleft  += (m_seq->get(hstart+i) == mouseSeq->get(mstart+i)) ? 1 : 0;
    if (humImg[hstop]  == mstop)  for (i=-2; i<=6; ++i) matchSSright += (m_seq->get(hstop+i)  == mouseSeq->get(mstop+i))  ? 1 : 0;
    score += matchSSleft  - 2;
    score += matchSSright - 4;
    break;
  case 1:
    if (humImg[hstop]  == mstop)  for (i=-2; i<=6; ++i) matchSSright += (m_seq->get(hstop+i)  == mouseSeq->get(mstop+i))  ? 1 : 0;
    score += matchSSright - 4;
    break;
  case 2:
    if (humImg[hstart] == mstart) for (i=-3; i<=0; ++i) matchSSleft  += (m_seq->get(hstart+i) == mouseSeq->get(mstart+i)) ? 1 : 0;
    score += matchSSleft  - 2;
    break;
  case 3:
    break;
  }
  if (VERBOSE)  cout << "\t ...: " << score << endl;
  */
  delete[] haa;
  delete[] maa;
  delete[] haaimg;
  delete[] maaimg;

  return score;
}


int Modules::exonImagesToTry(Modules *mouseModule, int hstart, int hstop, int *mstarts, int *mstops, int rType) {
  
  register int i,j;
  int mleftMap=-1, mrightMap=-1;
  int leftType  = (rType & 1) ? ATG        : START;
  int rightType = (rType & 2) ? STOP_CODON : STOP;
  int mseql = mouseModule->seql;

  for (i=hstart; i>=0; i--)
    if (humImg[i]>=0) {
      mleftMap = humImg[i];
      break;
    }
  for (i=hstop; i<=seql; i++)
    if (humImg[i]>=0) {
      mrightMap = humImg[i];
      break;
    }
  if (mleftMap == -1 || mrightMap == -1) return 0;
  assert(mleftMap < mrightMap);

  if (VERBOSE) cout << "\t\t\t\t\t mleftMap: " << mleftMap+mouseParseOffset << "\t mrightMap: " << mrightMap+mouseParseOffset << endl;
  
  int regCount = 4;
  for (i=0; i<4; ++i) mstarts[i] = mstops[i] = -1;

  i = mleftMap;
  do {
    if (mouseModule->endpointType[i] & leftType) {
      mstarts[0] = mstarts[1] = i;
      break;
    }
  } while (i-- > 0);
  i = mleftMap;
  do {
    if (mouseModule->endpointType[i] & leftType) {
      mstarts[2] = mstarts[3] = i;
      break;
    }
  } while (i++ < mseql);
  i = mrightMap;
  do {
    if (mouseModule->endpointType[i] & rightType) {
      mstops[1] = mstops[3] = i;
      break;
    }
  } while (i++ < mseql);

  //  if (rightType == STOP_CODON) i = mrightMap-1;
  //  else                         i = mrightMap;
  i = mrightMap;
  do {
    if (mouseModule->endpointType[i] & rightType) {
      mstops[0] = mstops[2] = i;
      break;
    }
  } while (i-- > 0);

  if (VERBOSE) {
    cout << "\t\t\t\t\t";
    for (i=0; i<4; ++i) cout << mstarts[i]+mouseParseOffset << "," << mstops[i]+mouseParseOffset << "; ";
    cout << endl;
  }

  int changed = 0;
  do {
    changed = 0;
    for (i=0; i<regCount; ++i) {
      if (mstarts[i] == -1 || mstops[i] == -1 || mstarts[i] >= mstops[i] || ABSDIFF(mstarts[i],mleftMap) > 21 || ABSDIFF(mstops[i],mrightMap) > 21 )
	if (i < regCount-1) { 
	  changed = 1; 
	  if (VERBOSE) cout << "\t\t\t\t\t\t discarding: " << mstarts[i]+mouseParseOffset << "," << mstops[i]+mouseParseOffset << endl;
	  for (j = i; j < regCount-1; ++j) { mstarts[j] = mstarts[j+1]; mstops[j] = mstops[j+1]; }
	  regCount--;
	  break; 
	}
	else { 
	  if (VERBOSE) cout << "\t\t\t\t\t\t throwing: " << mstarts[i]+mouseParseOffset << "," << mstops[i]+mouseParseOffset << endl;
	  changed = 1; 
	  regCount--; 
	  break; 
	}
      
      if (i < regCount-1 && (mstarts[i] == mstarts[i+1] && mstops[i] == mstops[i+1])) {
	changed = 1;
	if (VERBOSE) cout << "\t\t\t\t\t\t discard: " << mstarts[i]+mouseParseOffset << "," << mstops[i]+mouseParseOffset << endl;
	for (j = i; j < regCount-1; ++j) { mstarts[j] = mstarts[j+1]; mstops[j] = mstops[j+1]; }
	regCount--;
	if (VERBOSE) {
	  cout << "\t\t\t\t\t\t";
	  for (j=0; j<regCount; ++j) cout << mstarts[j]+mouseParseOffset << "," << mstops[j]+mouseParseOffset << "\t";
	  cout << endl;
	}
	break;
      }
    }
  } while (changed);
  
  for (i=0; i<regCount; ++i) {
    assert(mstarts[i] < mstops[i]);
    assert(mouseModule->endpointType[mstarts[i]] & leftType);
    assert(mouseModule->endpointType[mstops[i]]  & rightType);
  }
 
  return regCount;
}


Parse* Modules::mouseGeneratePartialParse(Modules *mouseModule, Registry *reg) {
  int minimumIntronLength    = atoi(reg->lookupVal("minimumIntronLength"));
  register int i,j,k;

  if (VERBOSE) cout << "Human parse offset: " << parseOffset << "\t Mouse parse offset: " << mouseParseOffset << endl;
  
  ParseAssumptions parseKind = (ParseAssumptions)atoi(reg->lookupVal("ParseAssumptions"));

  assert(parseKind == SINGLE_GENE || parseKind == PARTIAL_GENE || parseKind == MULTI_GENES);
    
  hcodonMat = new (double**)[4]; // build human codon matrix
  for (i=0; i<4; ++i) hcodonMat[i] = new (double*)[4];
  for (i=0; i<4; ++i)
    for (j=0; j<4; ++j)
      hcodonMat[i][j]=new double[4];
  
  mcodonMat = new (double**)[4]; // build mouse codon matrix
  for (i=0; i<4; ++i) mcodonMat[i] = new (double*)[4];
  for (i=0; i<4; ++i)
    for (j=0; j<4; ++j)
      mcodonMat[i][j]=new double[4];
  
  ifstream hfcodonmat("/data2/rosetta/tables/CodonUsage/Homsap.cod"); // file for human codon usage
  ifstream mfcodonmat("/data2/rosetta/tables/CodonUsage/Musmus.cod"); // 
  readCodonMatrix(hcodonMat,hfcodonmat);
  readCodonMatrix(mcodonMat,mfcodonmat);
  
  pamMat = new (double*)[22];
  for (i=0; i<22; ++i) pamMat[i] = new double[22];
  ifstream fmat("/data2/rosetta/tables/PAM/PAM20");
  readPamMatrix(pamMat,fmat);
  normalizePam(2,pamMat);

  currAlignPAM = new (double*)[22];
  for (i=0; i<22; ++i) currAlignPAM[i] = new double[22];

  ifstream fmatAlign("/data2/rosetta/tables/PAM/PAM20"); 
  readPamMatrix(currAlignPAM,fmatAlign);

  mouseSeq  = mouseModule->getSequence();
  assert(mouseSeql == mouseSeq->get_length());
  
  for (i=1; i<=seql; ++i) if (humImg[i] > mouseSeql) humImg[i] = -1;

  HashSparseMatrix<int>  *nonZeroMap  = new HashSparseMatrix<int>(endcnt,endcnt,-1);
  
  int mouseEndcnt = mouseModule->endcnt;

  SparseMatrix<int>      *mouseStopMatrix  = mouseModule->getStopMatrix();
  HashSparseMatrix<int>  *mouseNonZeroMap  = new HashSparseMatrix<int>(mouseEndcnt, mouseEndcnt, -1);

  for (i=0; i<stopMatrix->sizeNonZero(); ++i) {
    int row_i = stopMatrix->row(i), col_i = stopMatrix->col(i);
    assert(endptrsInv[row_i] != -1 && endptrsInv[col_i] != -1);
    nonZeroMap->addEntry(endptrsInv[row_i],endptrsInv[col_i],i);
  }
  for (i=0; i<mouseStopMatrix->sizeNonZero(); ++i) {
    int row_i = mouseStopMatrix->row(i), col_i = mouseStopMatrix->col(i);
    mouseNonZeroMap->addEntry(mouseModule->endptrsInv[row_i], mouseModule->endptrsInv[col_i], i);
  }

  double      ***sites        = new (double**)[endcnt];
  int         ****siteBptr    = new (int***)[endcnt];
  int         ***siteMousemap = new (int**)[endcnt];
  int         ***siteMousefr  = new (int**)[endcnt];

  for (i=0; i<endcnt; ++i) {
    sites[i]        = new (double*[6]);
    siteBptr[i]     = new (int**[6]);
    siteMousemap[i] = new (int*)[6];
    siteMousefr[i]  = new (int*)[6];
    for (j=0; j<6; ++j) {
      siteBptr[i][j]     = new int*[3];
      sites[i][j]        = new double[3];
      siteMousemap[i][j] = new int[3];
      siteMousefr[i][j]  = new int[3];
      for (k=0; k<3; ++k) {
	sites[i][j][k]                = -Infinity;
	siteBptr[i][j][k]             = new int[3];
	siteBptr[i][j][k][ENDPTR]     = -1;         // Invalid endpointer;
	siteBptr[i][j][k][ENDPTRTYPE] = -1;         // Invalid endpointer type;
	siteBptr[i][j][k][FRAME]      = -1;         // Invalid frame;
	siteMousemap[i][j][k]         = -1;         // Invalid position in mouse;
	siteMousefr[i][j][k]          = -1;         // Invalid mouse frame;
      }
    }
  }

  for (k=0; k<3; ++k) {
    sites[0][BEGINNING][k] = 0;
    siteMousemap[0][BEGINNING][k] = 1;
  }

  int fr,lastfr,maxframe,maxj,currType,lastType;
  double maxscore, score, currentScore;

  endptrType[0]              = BEGIN_SEQ;
  mouseModule->endptrType[0] = BEGIN_SEQ;

  for (i=1; i<endcnt-1; ++i) { // Main iteration - end of current region;
    if (endptrType[i]&ATG) {
      if (VERBOSE) cout << " ATG (" << i << ") at: " << parseOffset+endptrs[i] << endl;

      for (fr=0;fr<3;fr++) {
	sites[i][START_CODING][fr] = ( (fr == (endptrs[i]%3)) ? 0 : -Infinity );
	siteBptr[i][START_CODING][fr][ENDPTR]     = 0;
	siteBptr[i][START_CODING][fr][ENDPTRTYPE] = BEGINNING;
	siteBptr[i][START_CODING][fr][FRAME]      = 0;
      }
      if (parseKind == MULTI_GENES)
	for (j=1; j<i; ++j)
	  if (endptrType[j]&STOP_CODON && endptrs[i]-endptrs[j]>1000) // Minimum Intragenic length;
	    for (fr=0; fr<3; ++fr)
	      if (sites[j][STOP_CODING][fr] > MAX(sites[i][START_CODING][fr], MAX(1.0, (3000-(endptrs[i]-endptrs[j]))/100.0))) {
		sites[i][START_CODING][fr] = sites[j][STOP_CODING][fr] - MAX(1.0, (3000-(endptrs[i]-endptrs[j]))/100.0);
		siteBptr[i][START_CODING][fr][ENDPTR]      = j;
		siteBptr[i][START_CODING][fr][ENDPTRTYPE]  = STOP_CODING;
		siteBptr[i][START_CODING][fr][FRAME]       = (endptrs[j]-2)%3;
	      }	
    }
    if (endptrType[i]&START && parseKind != SINGLE_GENE) { // start of partial parse;
      if (VERBOSE) cout << "AG at: " << parseOffset+endptrs[i] << endl;
      for (fr = 0; fr<3; ++fr) {
	sites[i][AG][fr]                = 0;
	siteBptr[i][AG][fr][ENDPTR]     = 0;
	siteBptr[i][AG][fr][ENDPTRTYPE] = BEGINNING;
	siteBptr[i][AG][fr][FRAME]      = 0;
      }
    }
    if (endptrType[i]&START) { // Intron;
      if (VERBOSE) cout << " START (" << i << ") at: " << parseOffset+endptrs[i] << endl;
      
      // int img_i = humImg[endptrs[i]];
      
      maxscore = -Infinity;
      for (j=1; j<i; ++j) {
	if (!(endptrType[j] & STOP)) continue;
	if (VERBOSE) cout << " \t STOP (" << j << ") at: " << parseOffset+endptrs[j] << endl;

	//	int img_j = humImg[endptrs[j]];

	if (!(endptrs[j]+minimumIntronLength < endptrs[i])) {
	  if (j==0 && VERBOSE) {
	    verb("\t\t\t not good/ no map");
	  }
	  continue;
	}

	int mfrom = humImg[endptrs[j]], mto = humImg[endptrs[i]];
	int mptr = endptrs[j]; while (mfrom < 0 && mptr >= 1)    mfrom = humImg[mptr--];
	mptr     = endptrs[i]; while (mto   < 0 && mptr <= seql) mto   = humImg[mptr++];
	
	if (mto-mfrom+1 < minimumIntronLength) {
	  verb("intron too short in mouse");
	  continue;
	}

	score = 0;
	for (fr=0;fr<3;fr++)
	  for (lastfr = 0; lastfr<3; ++lastfr) {
	    if (VERBOSE) cout << "\t\t <fr,lastFr>  == <" << fr << "," << lastfr << ">" << endl;
	    currentScore = score;
	    currentScore += sites[j][   GT    ][lastfr];
	    
	    if ((lastfr == (endptrs[j]-2)%3) && isStopCodon(m_seq->get(endptrs[j]-2), m_seq->get(endptrs[j]-1), m_seq->get(endptrs[j])))
	      { // MOUSE MODIFICATION NEEDED;
		verb("\t\t\t\t is the end of a parse");
		continue;
	      }
	    
	    if (!(((lastfr+endptrs[i]-endptrs[j]+5)%3==fr)&& // Forces frame consistency; MOUSE MODIFICATION NEEDED;
		  (!((( lastfr == ((endptrs[j]-1)%3)) && (isStopCodon(m_seq->get(endptrs[j]-1), m_seq->get(endptrs[j]), m_seq->get(endptrs[i]))))  ||
		     (( lastfr ==  (endptrs[j]   %3)) &&  isStopCodon(m_seq->get(endptrs[j]),   m_seq->get(endptrs[i]), m_seq->get(endptrs[i]+1))))))) {
	      verb("\t\t\t\t frame inconsistent");
	      //currentScore =  -Infinity;
	      continue;
	    }
	    
	    if (currentScore > sites[i][AG][fr]) {  // Compares score with previous best;
	      if (VERBOSE) cout << " Updating sites[" << i << "][" << (int)AG
				<< "][" << fr << "] to : "
				<< currentScore << " , ->" << parseOffset+endptrs[j] << " ( [" << j << "][" << (int)GT << "][" 
				<< lastfr << "] )" << endl;
	      sites[i][AG][fr]                = currentScore;
	      siteBptr[i][AG][fr][ENDPTR]     = j;
	      siteBptr[i][AG][fr][ENDPTRTYPE] = GT;
	      siteBptr[i][AG][fr][FRAME]      = lastfr;
	    }
	  }
      }
    }
    
    if ((endptrType[i]&STOP)||(endptrType[i]&STOP_CODON)) { // Exon;
      if (VERBOSE) cout << " STOP (" << i << ") at: " << parseOffset+endptrs[i] << endl;

      int region, mouseRegion, stopCode, mouseStopCode;
            
      for (j=i-1; j>=0; --j) {
	int hstart = endptrs[j], hstop = endptrs[i];
	if (cumulativeInvalidPositions[hstop] > cumulativeInvalidPositions[hstart]) break;
	
	int crossesBoundary = 0;
	
	for (k=0; k<alignRegcnt-1; ++k)
	  if (hstart < aligningEnds[k] - 40 && hstop > aligningBegins[k+1] + 40) crossesBoundary = 1;
	if (crossesBoundary) break;
	
	for (int rType = 0; rType < 4; ++rType) {
	  
	  if (!(     (  ((rType&1) && (endptrType[j]&ATG))         ||  ( (!(rType&1)) && (endptrType[j]&START))  )      &&
		     (  ((rType&2) && (endptrType[i]&STOP_CODON))  ||  ( (!(rType&2)) && (endptrType[i]&STOP ))  )         ))
	    continue; // MOUSE MODIFICATION NEEDED;

	  if (VERBOSE) cout << "*Exon " << rType << ": " << endptrs[j]+parseOffset << "," << endptrs[i]+parseOffset << endl;

	  int mstarts[100], mstops[100], mexonCnt;
	  
	  mexonCnt = exonImagesToTry(mouseModule, hstart, hstop, mstarts, mstops, rType);

	  if (!mexonCnt && VERBOSE) cout << "\t no images to try" << endl;
	  
	  for (int mexon = 0; mexon < mexonCnt; ++mexon) {
	    int mstart = mstarts[mexon], mstop = mstops[mexon];
	    
	    if (mouseModule->cumulativeInvalidPositions[mstop] > mouseModule->cumulativeInvalidPositions[mstart]) continue;
	    
	    crossesBoundary = 0;
	    for (k = 0; k<alignRegcnt-1; ++k)
	      if (mstart < mouseModule->aligningBegins[k]-40 && mstop > mouseModule->aligningBegins[k+1]+40) crossesBoundary = 1;
	    if (crossesBoundary) continue;
	    

	    if (!(rType&2) && m_seq->get(hstop+2) == BASE_C)
	      if ((mouseSeq->get(mstop+2) != BASE_C)){
		cout << "GC splice site maps to non-GC splice site: REJECTED" << endl;
		continue;
	      }
	    
	    if (VERBOSE) cout << "\t " << rType << " maps to: " << mstart+mouseParseOffset << "," << mstop+mouseParseOffset << endl;

	    int mouse_j = mouseModule->endptrsInv[mstart], mouse_i = mouseModule->endptrsInv[mstop];
	    
	    if ((region=nonZeroMap->get(j,i))==-1) {
	      if (VERBOSE) cout << "\t\t invalid exon" << endl;
	      continue; // Invalid exon;
	    }
	    if (VERBOSE) cout << "\t START/ATG (" << j << ") at: " << parseOffset+hstart << endl;
	    
	    if ((mouseRegion=mouseNonZeroMap->get(mouse_j,mouse_i))==-1) 
	      {
		if (VERBOSE) {
		  cout << "\t\t\t Mouse Region: " << mstart+mouseParseOffset << "," << mstop+mouseParseOffset << "  is invalid\t";
		  cout << "endptrsInv: " << mouse_j << ":" << mouseModule->endptrType[mouse_j]
		       << "," << mouse_i << ":" << mouseModule->endptrType[mouse_i] << "\t"
		       << mouseStopMatrix->get(mouse_j,mouse_i) << endl;
		}
		continue; // Invalid exon in mouse;
	      }
	    assert(region < stopMatrix->sizeNonZero() && mouseRegion < mouseStopMatrix->sizeNonZero());
	    	    
	    if ( (rType == 0 && (hstop-hstart+1<15  || mstop-mstart+1<15))  ||
		 (rType == 3 && (hstop-hstart+1<300 || mstop-mstart+1<300)) ||
		 (rType == 2 && (hstop-hstart+1<15  || mstop-mstart+1<15))  ||
		 (rType == 1 && (hstop-hstart+1<15  || mstop-mstart+1<15)) ) {
	      verb("\t\t\t too short");
	      continue; // Exon too short;
	    }
	    
	    stopCode      = stopMatrix->get(region);
	    mouseStopCode = mouseStopMatrix->get(mouseRegion);

	    if (rType&1) {
	      stopCode      |= (1<<((hstart+1) % 3)) | (1<<((hstart+2) % 3));
	      mouseStopCode |= (1<<((mstart+1) % 3)) | (1<<((mstart+2) % 3));
	    }	    
	    if (rType&2) {
	      stopCode      |= (1<<(hstop % 3)) | (1<<((hstop-1) % 3));
	      mouseStopCode |= (1<<(mstop % 3)) | (1<<((mstop-1) % 3));
	    }
	    
	    for (fr=0;fr<3;fr++) {
	      currType     = (rType&2) ? STOP_CODING  : GT;
	      lastType     = (rType&1) ? START_CODING : AG;
	    
	      if (VERBOSE) cout << "\t\t Fr" << fr << " examining..." << endl;

	      if ((1<<fr)&stopCode) {
		verb("\t\t\t\t invalid frame");
		continue;  // Continue only if frame fr is possible;
	      }

	      for (int mouseFr = 0; mouseFr<3; ++mouseFr) {
		if (VERBOSE) cout << "\t\t\t mouseFr" << mouseFr << endl;
		if ((1<<mouseFr)&mouseStopCode) {
		  verb("\t\t\t\t invalid frame in mouse");
		  continue;  // Continue only if frame fr is possible in mouse;
		}

		// if (rType&1) score  = endpointATGScore[hstart]   + mouseModule->endpointATGScore[mstart];
		double SSscore;
		if (rType&1) score  = 1* ( endpointATGScore[hstart]       + mouseModule->endpointATGScore[mstart]);
		else         score  = endpointStartScore[hstart]*0.5 + mouseModule->endpointStartScore[mstart]*0.5;

		if (rType&2) score += 0* (stopScore(m_seq->get(hstop-1),m_seq->get(hstop), 1)
			       + stopScore(mouseSeq->get(mstop-1),mouseSeq->get(mstop), 0));
		else         score += endpointStopScore[hstop]*3.5   + mouseModule->endpointStopScore[mstop]*3.5;
		SSscore = score;
				
		switch (rType) {
		case 0: if (score < -5) { if (VERBOSE) cout << "\t\t\t\t bad splice sites: " << score << endl; continue; } break;
		case 1:
		case 2: if (score < -5) { if (VERBOSE) cout << "\t\t\t\t bad splice sites: " << score << endl; continue; } break;
		case 3: break;
		}
		if (VERBOSE) cout << "\t\t\t\t SS Score: " << score << endl;
		
		double alignScore = exonAlignmentScore(m_seq, mouseSeq, hstart, hstop, mstart, mstop, 2,-1,-3, 0.84, 1.2, 7, humImg, mouseImg, 1);
		if (VERBOSE) cout << "Align to " << mstart+mouseParseOffset << "," << mstop+mouseParseOffset << " ,\t score: " << alignScore << endl;
		if (alignScore/(hstop-hstart+1) < 0.5) score = -Infinity;

		double epscore = (hstop-hstart+1>=15) ? exonPairScore(hstart,hstop,fr, mstart, mstop, mouseFr, rType) : -Infinity;
		//		if (epscore<5) epscore=-Infinity;
		
		score += epscore;

		if (ltt) { // INTRON TUPLES !!!;
		  score -= 4* (intronSignals[MAX(hstart,hstop-11)] - intronSignals[hstart]) +
		    (mouseModule->intronSignals[MAX(mstart,mstop-11)] - mouseModule->intronSignals[mstart]);
		}
		if (VERBOSE) cout << "Exon Pair Score for " << hstart+parseOffset << "," << hstop+parseOffset << ": " << epscore << "\t total score: " << score << endl;
		
		if (VERBOSE) cout << "\t\t\t\t Final Score " << score << endl;
		currentScore = score + sites[j][lastType][fr];
		if (VERBOSE) {
		  cout << "\t\t\t\t\t lastType: " << lastType << "  frame: " << fr << endl;
		  cout << "\t\t\t\t\t Current Score " << currentScore << "  LastScore " << sites[i][currType][fr] << endl;
		}

		{
		  int prevExonStopPtr  = siteBptr[j][lastType][fr][ENDPTR];
		  int prevExonStopType = siteBptr[j][lastType][fr][ENDPTRTYPE];
		  int prevExonFr       = siteBptr[j][lastType][fr][FRAME];
		  assert(prevExonStopPtr==0 || prevExonStopType&STOP || prevExonStopType&BEGIN_SEQ || (parseKind == MULTI_GENES && prevExonStopType&STOP_CODING));
		  if (prevExonStopType != -1 && prevExonStopType&STOP) {
		    assert( ( prevExonFr + (endptrs[j] - endptrs[prevExonStopPtr] - 1) )%3 == fr);
		    int prevExonMstop = siteMousemap[prevExonStopPtr][prevExonStopType][prevExonFr];
		    int prevExonMfr   = siteMousefr[prevExonStopPtr][prevExonStopType][prevExonFr];
		  
		    if ( (prevExonMfr + (mstart - prevExonMstop - 1) )%3 != mouseFr) {  // Mouse frame consistency;
		      if (VERBOSE) cout << "Frame inconsistent with previous exon in mouse" << endl;
		      continue;
		    }
		    int prevExonStartPtr  = siteBptr[prevExonStopPtr][prevExonStopType][prevExonFr][ENDPTR];
		    int prevExonStartType = siteBptr[prevExonStopPtr][prevExonStopType][prevExonFr][ENDPTRTYPE];
		    
		    int prevExonMstart    = siteMousemap[prevExonStartPtr][prevExonStartType][prevExonFr];
		    
		    int bigregion      = nonZeroMap->get(prevExonStartPtr, i);
		    int bigmouseRegion = mouseNonZeroMap->get(mouseModule->endptrsInv[prevExonMstart], mouse_i);
		    
		    if (bigregion != -1 && bigmouseRegion != -1 &&
			cumulativeInvalidPositions[hstop] == cumulativeInvalidPositions[endptrs[prevExonStartPtr]] &&
			mouseModule->cumulativeInvalidPositions[mstop] == mouseModule->cumulativeInvalidPositions[prevExonMstart]) {
		      int bigstopCode      = stopMatrix->get(bigregion);
		      int bigmouseStopCode = mouseStopMatrix->get(bigmouseRegion);
		      if ( !((1<<fr)&bigstopCode) && !((1<<mouseFr)&bigmouseStopCode) ) {
			if (VERBOSE) cout << "\t\t PENALTY FOR INTRON THAT COULD BE REMOVED: " << endptrs[prevExonStartPtr] << ","  << hstop << endl;
			currentScore -= MAX(2.0,SSscore);
		      }
		    }
		  }
		}
		if (currentScore>sites[i][currType][fr]) {
		  
		  sites[i][currType][fr] = currentScore;
		  siteBptr[i][currType][fr][ENDPTR]     = j;
		  siteBptr[i][currType][fr][ENDPTRTYPE] = lastType;
		  siteBptr[i][currType][fr][FRAME]      = fr;
		  
		  siteMousemap[j][lastType][fr]         = mstart;
		  siteMousemap[i][currType][fr]         = mstop;

		  siteMousefr[j][lastType][fr]          = mouseFr;
		  siteMousefr[i][currType][fr]          = mouseFr;

		  if (VERBOSE) {
		    cout << "\t\t\t\t\t Updating sites[" << i << "][" << currType << "][" 
			 << fr << "] to : " << currentScore << " , ->" << parseOffset+endptrs[j]
			 <<  " ( [" << j << "][" << lastType 
			 << "][" << fr << "] )" << endl;
		  }
		}
	      }
	    }
	  }
	}
      }
    }  
  }
  maxscore=-0.0123456789;
  maxframe=0;
  maxj=0;
  for(i=0;i<endcnt;i++) 
    for(fr=0;fr<3;fr++) {
      if (sites[i][GT][fr] > maxscore && parseKind != SINGLE_GENE) {
	if (VERBOSE) cout << "Updating final parse to: sites[" << i << "][GT][" << fr << "]    with score " << sites[i][GT][fr] << endl;
	maxscore = sites[i][GT][fr];
	currType = GT;
	maxj = i;
	maxframe = fr;
      }      
      if (sites[i][STOP_CODING][fr] > maxscore) {
	if (VERBOSE) cout << "Updating final parse to: sites[" << i << "][STOP_CODING][" << fr << "]    with score " << sites[i][STOP_CODING][fr] << endl;
	maxscore=sites[i][STOP_CODING][fr];
	currType = STOP_CODING;
	maxj=i;
	maxframe=fr;
      }
    }
  
  // Convert to special parse format
  i=maxj;
  fr=maxframe;
  
  Parse* parse=new Parse;
    
  while (i>0) {
    lastType = siteBptr[i][currType][fr][ENDPTRTYPE];
    j        = siteBptr[i][currType][fr][ENDPTR];
    if (( (lastType == START_CODING) || (lastType == AG))  &&
	( (currType == STOP_CODING ) || (currType == GT))) {
      parse->push_back(nonZeroMap->get(j,i));
    }
    fr       = siteBptr[i][currType][fr][FRAME];
    currType = lastType;
    i        = j;
  }

  if (VERBOSE)  cout <<  "Finished parse!  Score: " << maxscore << endl;
    
  for (i=0; i<endcnt; ++i) {
    for (j=0; j<6; ++j) {
      for (k=0; k<3; ++k) 
	delete[] siteBptr[i][j][k];
      delete[] siteMousemap[i][j];
      delete[] siteMousefr[i][j];

      delete[] sites[i][j];
      delete[] siteBptr[i][j];
    }
    delete[] siteMousemap[i];
    delete[] siteMousefr[i];

    delete[] sites[i];
    delete[] siteBptr[i];
  }
  delete[] siteMousemap;
  delete[] siteMousefr;
  
  delete[] sites;
  delete[] siteBptr;
  
  for (i=0; i<4; ++i) {
    for (j=0; j<4; ++j) {
      delete[] hcodonMat[i][j];
      delete[] mcodonMat[i][j];
    }
    delete[] hcodonMat[i];
    delete[] mcodonMat[i];
  }
  delete[] hcodonMat;
  delete[] mcodonMat;
  
  for (i=0; i<22; ++i){
    delete[] pamMat[i];
    delete[] currAlignPAM[i];
  }
  delete[] pamMat;
  delete[] currAlignPAM;
  
  delete nonZeroMap;
  delete mouseNonZeroMap;
  
  return parse;
}


char* Modules::sequenceSplice(char *seqText){
  char* seqSpliced = new char[codingl+1];
  int totsofar=0;
  for (int regionum = 1; regionum<=m_seq->get_region_num(); ++regionum) {
    Region *r = m_seq->get_region(regionum);
    if (r->type == REGION_CEXON) {
      strncpy(seqSpliced+totsofar,seqText+r->start-1,r->length());
      totsofar+=r->length();
    }
  }
  seqSpliced[codingl]='\0';
  return seqSpliced;
}

char* Modules::sequenceToChar(bool complement) {
  // The purpose of this function is to return an array which contains the sequence
  // masked for repeats and unwanted regions.  
  // complement = FALSE leaves the sequence as is.
  // complement = TRUE complements the sequence.
  
  int i;
  char* seqText           = new char[seql+1];

  seqText[0] = 'n';
  if (complement==TRUE){
    for(i=1;i<=seql;i++) {
      switch(m_seq->get(i)) {
      case BASE_A: seqText[i-1]='t'; break;
      case BASE_C: seqText[i-1]='g'; break;
      case BASE_G: seqText[i-1]='c'; break;
      case BASE_T: seqText[i-1]='a'; break;
      default: seqText[i-1]='n';
      }
    }
  }
  else {
     for(i=1;i<=seql;i++) {
       switch(m_seq->get(i)) {
       case BASE_A: seqText[i-1]='a'; break;
       case BASE_C: seqText[i-1]='c'; break;
       case BASE_G: seqText[i-1]='g'; break;
       case BASE_T: seqText[i-1]='t'; break;
       default: seqText[i-1]='n';
       }
     }
  }

  for (i=1; i<=seql; ++i) if (!validPositions[i]) seqText[i-1] = 'n';
   
   seqText[seql]='\0';
   return seqText;
}

int j_minus_7_scheme(int k) {
  
  if (k == 4)  return  5; //  3;
  if (k == 5)  return  6; //  3;
  if (k == 6)  return  6; //  5;
  if (k == 7)  return  6; //  6;
  if (k == 8)  return  6; //  8;
  if (k <= 10) return  8; //  9;
  else         return  8; // 10;
}

void Modules::computeDictionaryInfo(Registry *reg) {
  if (reg->lookupVal("errorMessage").length()) return;
  
  int proteinHomology = reg->lookupVal("proteinHomology").contains("yes");
  int mouseHomology   = reg->lookupVal("generateParse").contains("mouse");
  int cDNAHomology    = reg->lookupVal("cDNAHomology").contains("yes");
  
  int i,j;

  if (proteinHomology||cDNAHomology||mouseHomology) {
    cout << "Computing dictionary information..." << endl;
    
    if (mouseHomology) {
      cout << "MOUSE HOMOLOGY" << endl;
    }
    
    if ((proteinHomology||cDNAHomology) && (!mouseHomology)) {
      
      Dictionary *owl;
      char errMsg[128];
      errMsg[0]='\0';
      	
      verb("Creating a new dictionary...");
      owl=new Dictionary ("/data2/rosetta/dictionaries/dbOWL",errMsg);
      verb("done!");
      if (strlen(errMsg)) cout << "Dictionary error: " << errMsg << endl;
      else {
	char *seqText;
	seqText=sequenceToChar(FALSE);
	char *codingseqText;
	codingseqText=sequenceSplice(seqText);
	
	if (proteinHomology){
	  char *proteinText;
	  long unsigned int *protein, *proteinMaxSeglen;
	  vector<String> names;
	  ivector<long unsigned int> proteinSegpos;
	  ivector<long unsigned int> proteinSeglen;
	  ivector<long unsigned int> proteinSegacc;
	  ivector<long unsigned int> proteinImage;
	  char* proteinNam;
	  proteinText=new char[seql+3];
	  arrayInit(' ',proteinText,seql+3);
	  char* temp;
	  strcpy(proteinText,                DNAToProtein0 = m_seq->DNAToProtein(seqText));
	  strcpy(proteinText+(seql/3)+1,     DNAToProtein1 = m_seq->DNAToProtein(seqText+1));
	  strcpy(proteinText+(2*(seql/3))+2, DNAToProtein2 = m_seq->DNAToProtein(seqText+2));
	  protein          = new (long unsigned int)[seql+3]; arrayZero(protein,seql+3);
	  proteinMaxSeglen = new (long unsigned int)[seql+3]; arrayZero(proteinMaxSeglen,seql+3);
	  int proteinCutoff=atoi(reg->lookupVal("proteinCutoff"));
	  String proteinAll=reg->lookupVal("proteinAll");
	  
	  cout << "Finding protein segments..." << endl;
	  //	  cout << "PROTEIN TEXT: " << endl << proteinText << endl << endl;

	  owl->Segments(proteinText,seql+3,proteinCutoff,proteinSegpos, proteinImage, proteinSeglen, proteinSegacc);
	  int segcount      = proteinSegpos.size();
	  int finalSegCount = 0;
	  cout << "found " << segcount << " segments." << endl;
	  int *pSegposArray = new int[segcount]; arrayZero(pSegposArray,segcount);
	  int *pSeglenArray = new int[segcount]; arrayZero(pSeglenArray,segcount);
	  int *pSegaccArray = new int[segcount]; arrayZero(pSegaccArray,segcount);
	  int *pImageArray  = new int[segcount]; arrayZero(pImageArray, segcount);
	  
	  if (reg->lookupVal("proteinMinusOne").contains("yes")) {
	    ivector<long unsigned int> codingSegpos;
	    ivector<long unsigned int> codingSeglen;
	    ivector<long unsigned int> codingSegacc;
	    ivector<long unsigned int> codingImage;

	    char *codingText = new char[seql+3];
	    strcpy(codingText, temp = m_seq->DNAToProtein(codingseqText));

	    int codingCutoff = MAX(codingl/3-12, codingl/4);
	    cout << "Finding coding segments..." << endl;
	    owl->Segments(codingText,codingl/3,codingCutoff,codingSegpos, codingImage, codingSeglen, codingSegacc);

	    for (i=0; i<(int)codingSegacc.size(); ++i) proteinAccessions[codingSegacc[i]] = 1;
	  }
	  
	  for (i=0; i<segcount; ++i) {
	    if (proteinAccessions[proteinSegacc[i]]==1) {
	      cout << "Removing: " << proteinSegpos[i] << " , " << proteinSeglen[i] << " , " << proteinSegacc[i] << endl;
	      continue;
	    }
	    pSegposArray[finalSegCount] = proteinSegpos[i];
	    pSeglenArray[finalSegCount] = proteinSeglen[i];
	    pSegaccArray[finalSegCount] = proteinSegacc[i];
	    pImageArray[finalSegCount]  = proteinImage[i];
	    finalSegCount++;
	  }
	  cout << "Minus one removes " << segcount-finalSegCount << " segments" << endl;

	  cout << "Finding hits..." << endl;
	  if (proteinAll.contains("every")) {
	    cout << "running this body of code" << endl;
	    orWrapper(protein, pSegposArray, pSeglenArray, finalSegCount);
	    for (i=0; i<finalSegCount; ++i)
	      for (j=pSegposArray[i]; j<pSegposArray[i] + pSeglenArray[i]; ++j)
		proteinMaxSeglen[j] = max(pSeglenArray[i], (int)proteinMaxSeglen[j]);
	    
	    // The proteinMinusOne option removes an exon from the hits if there is a tuple hit
	    // that matches it completely.
	    
	    if (proteinMethod==3) {
	      FILE *ranks5 = fopen("rank.five","r");
	      double tsz5 = 20*20*20*20*20;
	      for (i=1; i<seql; ++i) {
		double prob = 0;
		switch (i%3) {
		case 0: 
		  proteinCumulativeLogs[i +   seql + 1] = proteinCumulativeLogs[i-1 +   seql + 1];
		  proteinCumulativeLogs[i + 2*seql + 2] = proteinCumulativeLogs[i-1 + 2*seql + 2];

		  prob = (double)tupfreq(DNAToProtein0 + i/3, 5, ranks5) / tsz5;
		  
		  proteinCumulativeLogs[i]              = proteinCumulativeLogs[i-1] + prob * log(prob);
		  break;
		case 1:

		  prob = (double)tupfreq(DNAToProtein1 + i/3, 5, ranks5) / tsz5;

		  proteinCumulativeLogs[i +   seql + 1] = proteinCumulativeLogs[i-1 +   seql + 1] + prob * log(prob);
		  proteinCumulativeLogs[i + 2*seql + 2] = proteinCumulativeLogs[i-1 + 2*seql + 2];
		  proteinCumulativeLogs[i]              = proteinCumulativeLogs[i-1];
		  break;
		case 2:
		  proteinCumulativeLogs[i +   seql + 1] = proteinCumulativeLogs[i-1 +   seql + 1];
		  
		  prob = (double)tupfreq(DNAToProtein2 + i/3, 5, ranks5) / tsz5;
		  
		  proteinCumulativeLogs[i + 2*seql + 2] = proteinCumulativeLogs[i-1 + 2*seql + 2] + prob * log(prob);
		  proteinCumulativeLogs[i]              = proteinCumulativeLogs[i-1];
		  break;
		}
	      }
	      fclose(ranks5);
	    }

	    if (1==0) {
	      int numtwo=0;
	      int frametwo=0;
	      cout << "Checking for alternative splice sites" << endl;
	      for (int regionum=1; regionum<=m_seq->get_region_num();regionum++){
		Region *r = m_seq->get_region(regionum);
		if (r->type==REGION_CEXON && r->length()>=30){
		  frametwo=0;
		  for (int proteinPiece=0; proteinPiece < finalSegCount; proteinPiece++){
		    for (int proteinPiece2=0; proteinPiece2 < finalSegCount; proteinPiece2++){
		      long int start1 = (pSegposArray[proteinPiece] % ((seql/3)+1))*3,
			start2 = (pSegposArray[proteinPiece2] % ((seql/3)+1))*3;
		      long int stop1 = start1 + pSeglenArray[proteinPiece]*3,
			stop2 = start2 + pSeglenArray[proteinPiece2]*3;
		      int rlen = r->length();
		     
		      if ( MIN(stop1, r->stop) - MAX(start1, r->start) >= MIN(100, MAX(rlen/2, 24)) &&
			   MIN(stop2, r->stop) - MAX(start2, r->start) >= MIN(100, MAX(rlen/2, 24)) &&
			   MIN(stop1, stop2)   - MAX(start1, start2)   >= MIN(100, MAX(rlen/2, 24)) &&
			   (floor(pSegposArray[proteinPiece]/((seql/3)+1)) != floor(pSegposArray[proteinPiece2]/((seql/3)+1)))) {
			frametwo=1;
		      }
		    }
		  }
		  if (frametwo) {
		    numtwo++;
		    infoOut << "Sequence where exon with two frames occurs: " << m_seq->get_locus() << endl;
		    infoOut << "Exon positions: " << r->start << "," << r->stop << endl;
		  }
		}
	      }
	      myData[TWO_FRAME_EXONS] = numtwo;
	    }
	    cout << finalSegCount << " Protein segments found" << endl;
	    
	    if (proteinCutoff >= 7) {
	      // Begin creating html file:
	      
	      htmlFile.push_back("<b>PROTEIN MATCHES:</b>\n<pre>\n\n</pre>\n<table border=\"1\" width=\"100%\">\n<tr><td align=\"center\">Position</td><td align=\"center\">Length</td><td align=\"center\">Frame</td><td align=\"center\">Locus</td></tr>"); 
	      
	      for(int j=0;j<finalSegCount;j++) {
		proteinNam = new char[100];
		owl->getSequenceName(pSegaccArray[j],proteinNam);
		String proteinNameString = proteinNam;
		delete[] proteinNam;
		
		htmlFile.push_back("<tr><td align=\"center\">"+toa((pSegposArray[j])%((seql/3)+1)*3)+
				   "</td><td align=\"center\">"+toa((pSeglenArray[j])*3)+"</td><td align=\"center\">"+
				   toa(floor(pSegposArray[j]/((seql/3)+1)))+
				   "</td><td align=\"center\"><a href=\"http://bmbsgi11.leeds.ac.uk/cgi-bin/owl.sh?d%2Ffull+code+%22"+
				   toa(proteinNameString)+"%22\">"+toa(proteinNameString)+"</a></td></tr>\n");	    
	      }
	      htmlFile.push_back("</table>\n");
	    }
	    // Begin Process of placing the protein information in proteinHits:
	  }

	  int i,j;
	  for(i=0;i<((seql/3)+1);i++)
	    for(j=1;j<4;j++) {
	      if ((3*i+j)<=seql) 
		proteinHits[3*i+j]=protein[i];
	      if ((3*i+j+1)<=seql)
		proteinHits[3*i+j+seql+1]=protein[i+(seql/3)+1];
	      if ((3*i+j+2)<=seql)
		proteinHits[3*i+j+2*seql+2]=protein[i+2*(seql/3)+2];
	    }

	  for(i=0;i<((seql/3)+1);i++) 
	    for(j=1;j<4;j++) {
	      if ((3*i+j)<=seql) 
		proteinSeglenHits[3*i+j]=MAX((int)proteinMaxSeglen[i], 4);
	      if ((3*i+j+1)<=seql)
		proteinSeglenHits[3*i+j+seql+1]=MAX((int)proteinMaxSeglen[i+(seql/3)+1],4);
	      if ((3*i+j+2)<=seql)
		proteinSeglenHits[3*i+j+2*seql+2]=MAX((int)proteinMaxSeglen[i+2*(seql/3)+2],4);
	    }
	  if (reg->lookupVal("generateParse").contains("protein")){
	    int intronSignal_Covered = 0;
	    
	    for (i=1; i<=seql; ++i)
	      for (j=0; j<3; ++j) {

		proteinCumulativeHits[i + j*(seql+1)] = 
		  proteinCumulativeHits[i-1 + j*(seql+1)]+proteinHits[i + j*(seql+1)];
		
		proteinCumulativeSeglenHits[i + j*(seql+1)] = 
		  proteinCumulativeSeglenHits[i-1 + j*(seql+1)]+ j_minus_7_scheme(proteinSeglenHits[i + j*(seql+1)]);
		
		if (intronSignals[i]-intronSignals[i-1]>0) intronSignal_Covered = 12;
		else                                       intronSignal_Covered--;
		
		if (intronSignal_Covered <= 0) {
		  proteinMaskedIntronsCumulativeHits[i + j*(seql+1)] =
		    proteinMaskedIntronsCumulativeHits[i-1 + j*(seql+1)] + proteinHits[i + j*(seql+1)];
		}
		else {
		  proteinMaskedIntronsCumulativeHits[i + j*(seql+1)] =
		    proteinMaskedIntronsCumulativeHits[i-1 + j*(seql+1)];
		}
		proteinCumulativeHitPositions[i + j*(seql+1)] =
		  proteinCumulativeHitPositions[i-1 + j*(seql+1)] +
		  (proteinHits[i + j*(seql+1)]>0 ? 1 : 0);
	      }
	    for (i=1; i<=m_seq->get_region_num(); i++) {
	      Region *r = m_seq->get_region(i);
	      if (r->type==REGION_CEXON)
		for (j=r->start; j<=r->stop; ++j) {
		  if (proteinHits[j] || proteinHits[j + (seql+1)] || proteinHits[j + 2*(seql+1)]) myData[EXON_POS_COVERED]+=1;
		  else myData[EXON_POS_UNCOV]+=1;
		}
	      
	      if (r->type==REGION_INTRON)
		for (j=r->start; j<=r->stop; ++j) {
		  if (proteinHits[j] || proteinHits[j + (seql+1)] || proteinHits[j + 2*(seql+1)]) myData[INTRON_POS_COVERED]+=1;
		  else myData[INTRON_POS_UNCOV]+=1;
		}
	    }
	    cout << "Exon positions covered: " << myData[EXON_POS_COVERED] << endl;
	    cout << "Intron positions covered: " << myData[INTRON_POS_COVERED] << endl;
	  }
	  delete[] pSegposArray;
	  delete[] pSeglenArray;
	  delete[] pSegaccArray;
	  delete[] pImageArray;
	  delete[] protein;
	  delete[] proteinText;	  
	}
	if (cDNAHomology) {
	  // First we build cDNAtext which contains the sequence in 4 orientations
	  // The order of the orientations is forward, reverse, forward complement
	  // and reverse complement.
	  
	  Dictionary *est;
	  verb("Creating a new cDNA dictionary...");
	  est=new Dictionary ("/data/tmp/repeats.val4",errMsg);
	  verb("done!");  
	  if (strlen(errMsg)) cout << "Dictionary error: " << errMsg << endl;
	  
	  long unsigned int *cDNAMatches;
	  cDNAMatches=new (long unsigned int)[4*seql+3];
	  arrayZero(cDNAMatches,4*seql+3);
	  char* cDNANam;
	  ivector<long unsigned int> cDNASegpos;
	  ivector<long unsigned int> cDNASeglen;
	  ivector<long unsigned int> cDNASegacc;
	  ivector<long unsigned int> cDNAImage; 
	  int cDNACutoff=atoi(reg->lookupVal("cDNACutoff"));
	  char *cDNAtext;
	  cDNAtext = new char[seql*4+3];
	  arrayInit(' ',cDNAtext,seql*4+3);
	  char *seqTextComplement;
	  seqTextComplement = sequenceToChar(TRUE);
	  for (i=0; i<seql; ++i){
	    cDNAtext[i]          = seqText[i];
	    cDNAtext[seql+i+1]   = seqText[seql-i-1];
	    cDNAtext[2*seql+i+2] = seqTextComplement[i];
	    cDNAtext[3*seql+i+3] = seqTextComplement[seql-i-1];
	  }	  
	  
	  cout << "Finding cDNA segments..." << endl;
	  est->Segments(cDNAtext,seql*4+3,cDNACutoff,cDNASegpos, cDNAImage, cDNASeglen, cDNASegacc);
	  int finalCSegCount = cDNASegpos.size();
	  
	  int *cSegposArray = new int[finalCSegCount];
	  int *cSeglenArray = new int[finalCSegCount];
	  int *cSegaccArray = new int[finalCSegCount];
	  int *cImageArray  = new int[finalCSegCount];
	  for (i=0; i<finalCSegCount; ++i) {
	    cSegposArray[i] = cDNASegpos[i];
	    cSeglenArray[i] = cDNASeglen[i];
	    cSegaccArray[i] = cDNASegacc[i];
	    cImageArray[i]  = cDNAImage[i];
	  }

	  cout << "Finding hits..." << endl;
	  if (reg->lookupVal("cDNAAll").contains("every"))
	    orWrapper(cDNAMatches, cSegposArray, cSeglenArray, finalCSegCount);
	  //cDNAMatches is 4 times the length, it needs to be converted into cDNAHits
	  for (int i=0; i<seql; i++)
	    cDNAHits[i]=cDNAMatches[i]+cDNAMatches[2*seql-i]+cDNAMatches[2*seql+i+2]+cDNAMatches[4*seql-i+2];
	  
	  // Begin creating html file:
	  
	  htmlFile.push_back("<p><b>cDNA MATCHES:</b>\n<pre>\n\n</pre>\n<table border=\"1\" width=\"100%\">\n<tr><td align=\"center\">Position</td><td align=\"center\">Length</td><td align=\"center\">Orientation</td><td align=\"center\">Locus</td></tr>"); 
	  cout << cDNASegpos.size() << " cDNA segments found" << endl;
	  
	  for(int j=0;j<(int)cDNASegpos.size();j++) {
	    cDNANam = new char[100];
	    est->getSequenceName(cDNASegacc[j],cDNANam);
	    String cDNANameString = cDNANam;
	    delete[] cDNANam;
	    
	    htmlFile.push_back("<tr><td align=\"center\">"+toa((cDNASegpos[j])%(seql))+"</td><td align=\"center\">"+toa((cDNASeglen[j]))+"</td><td align=\"center\">"+toa(floor(cDNASegpos[j]/(seql)))+"</td><td align=\"center\"><a href=\"http://www.ncbi.nlm.nih.gov/htbin-post/Entrez/query?db=n&form=6&uid="+toa(cDNANameString)+"&dopt=g\">"+toa(cDNANameString)+"</a></td></tr>\n");	 
	  }
	  htmlFile.push_back("</table>\n");
	  
	  if (reg->lookupVal("generateParse").contains("cDNA")){
	    for (int i=1; i<=seql; ++i)
	      for (int j=0; j<3; ++j) {
		proteinCumulativeHits[i + j*(seql+1)] =
		  proteinCumulativeHits[i-1 + j*(seql+1)]+cDNAHits[i-1];
		proteinCumulativeHitPositions[i + j*(seql+1)] =
		  proteinCumulativeHitPositions[i-1 + j*(seql+1)] +
		  (cDNAHits[i-1] ? 1 : 0);
	      }   
	    for (int i=1; i<=m_seq->get_region_num(); i++) {
	      Region *r = m_seq->get_region(i);
	      if (r->type==REGION_CEXON)
		for (int pos=r->start; pos<=r->stop; ++pos) {
		  if (proteinHits[pos] || proteinHits[pos + (seql+1)] || proteinHits[pos + 2*(seql+1)]) myData[EXON_POS_COVERED]+=1;
		  else myData[EXON_POS_UNCOV]+=1;
		}
	      if (r->type==REGION_INTRON)
		for (int pos=r->start; pos<=r->stop; ++pos) {
		  if (proteinHits[pos] || proteinHits[pos + (seql+1)] || proteinHits[pos + 2*(seql+1)]) myData[INTRON_POS_COVERED]+=1;
		  else myData[INTRON_POS_UNCOV]+=1;
		}
	    }
	  }
	  delete[] cSegposArray;
	  delete[] cSeglenArray;
	  delete[] cSegaccArray;
	  delete[] cImageArray;
	  delete[] cDNAMatches;
	  delete[] cDNAtext;
	  delete[] seqTextComplement;
	  delete est;
	}
	delete[] seqText;
	delete[] codingseqText;
      }
      delete owl;
    }
  }
}

void Modules::outputParse(Parse *parse) {
  
  SparseMatrix<double>  *scorem = getScoreMatrix();
  int plength   = parse->size();
  int pregions[1000];
  int prstarts[1000];
  int prstops[1000];
  int prframes[1000];

  for (int i=0; i<plength; ++i) {
    pregions[i] = (*parse)[plength-1-i];
    prstarts[i] = scorem->row(pregions[i]);
    prstops[i]  = scorem->col(pregions[i]);

    prframes[i] =  ( i==0 )  ?  prstarts[0] % 3  :  ( 3+ prframes[i-1] + prstarts[i] - prstops[i-1] - 1 ) %3;
  }

  cout << "PARSE:" << endl;
  for(int x=0; x<plength; ++x) {
    int start   = prstarts[x] + parseOffset;
    int stop    = prstops[x]  + parseOffset;
    int myFrame = prframes[x];

    cout << " (" << start << "," << stop << ")\t" << myFrame << endl;
  }
}

void Modules::mouseOutputParse(Parse *parse, ofstream  &fout) {
  
  SparseMatrix<int>  *stopm = getStopMatrix();
  int plength   = parse->size();
  int pregions[1000];
  int prstarts[1000];
  int prstops[1000];
  int prframes[1000];

  for (int i=0; i<plength; ++i) {
    pregions[i] = (*parse)[plength-1-i];
    prstarts[i] = stopm->row(pregions[i]);
    prstops[i]  = stopm->col(pregions[i]);

    prframes[i] =  ( i==0 )  ?  prstarts[0] % 3  :  ( 3+ prframes[i-1] + prstarts[i] - prstops[i-1] - 1 ) %3;
  }

  if (VERBOSE)  cout << "PARSE:" << endl;
  for(int x=0; x<plength; ++x) {
    int start   = prstarts[x] + parseOffset;
    int stop    = prstops[x]  + parseOffset;
    int myFrame = prframes[x];

    if (VERBOSE)    cout << " (" << start << "," << stop << ")\t" << myFrame << endl;
    fout << start << "\t" << stop << "\t" << myFrame << endl;
  }
}


void Modules::mouseOutputParseRevComp(Parse *parse, ofstream  &fout) {
  
  SparseMatrix<int>  *stopm = getStopMatrix();
  int plength   = parse->size();
  int pregions[1000];
  int prstarts[1000];
  int prstops[1000];
  int prframes[1000];

  for (int i=0; i<plength; ++i) {
    pregions[i] = (*parse)[plength-1-i];
    prstarts[i] = stopm->row(pregions[i]);
    prstops[i]  = stopm->col(pregions[i]);

    prframes[i] =  ( i==0 )  ?  prstarts[0] % 3  :  ( 3+ prframes[i-1] + prstarts[i] - prstops[i-1] - 1 ) %3;
  }

  if (VERBOSE)  cout << "PARSE Reverse Complement:" << endl;
  for(int x=0; x<plength; ++x) {
    int start   = prstarts[x] + parseOffset;
    int stop    = prstops[x]  + parseOffset;
    int myFrame = prframes[x];

    if (VERBOSE)    cout << " (" << seql - start + 1 << "," << seql - stop + 1 << ")\t" << myFrame << endl;
    fout << seql - start + 1 << "\t" << seql - stop + 1 << "\t" << myFrame << endl;
  }
}


void Modules::outputParse(Parse *parse, Registry *reg) {

  int trash=0;
  int j;
  vector<int>    *trashArray  = new vector<int>;
  vector<double> *trashArray2 = new vector<double>;
  vector<int>    *trashArray3 = new vector<int>;
  for(j=0; j<20; j++) trashArray->push_back(0);
  for(j=0; j<20;  j++) trashArray2->push_back(0);
  for(j=0; j<20;  j++) trashArray3->push_back(0);
  outputParse(parse,reg, trash,trash,trash,trash,trash,trash,trash,trashArray3,
	      trashArray,trashArray2); 
  delete trashArray;
  delete trashArray2;
  delete trashArray3;
}

void Modules::outputParse(Parse *parse, 
			  Registry *reg,
			  int& argTotalOverlap, int& argTotalNonOverlap, 
			  int& argTotalPerfect, int& argTotalMatch,
			  int& argTotalMiss,    int& argTotalExons,
			  int& argTotalPExons,        vector<int> *argNucData,
			  vector<int> *argTotalFrame, vector<double> *argGCInfo) {

  // 0) PP 1) PN 2) AP 3) AN 4) TP 5) FP 6) TN 7) FN

  SparseMatrix<int>     *framem = getFrameMatrix();
  SparseMatrix<double>  *scorem = getScoreMatrix();
  cout << endl;
  cout << "Matches: " << endl;
  int totalOverlap    = 0;
  int totalNonOverlap = 0;
  int totalPerfect    = 0;
  int totalMatch      = 0;
  int totalExons      = 0;
  int totalPExons     = parse->size();
  int f,i;

  vector<double> *GCInfo = new vector<double>;
  for(f=0;f<8;f++) GCInfo->push_back(0);
  for(f=1;f<=seql;f++) {
    switch(m_seq->get(f)) {
    case BASE_A: { (*GCInfo)[0]++; break; }
    case BASE_C: { (*GCInfo)[1]++; break; }
    case BASE_G: { (*GCInfo)[2]++; break; }
    case BASE_T: { (*GCInfo)[3]++; break; }
    default: ;
    }
  }
  for(f=0;f<4;f++) (*GCInfo)[f+4]=( ((double)((*GCInfo)[f])) * ( ((double)endcnt) / ((double)(seql)) ) );
  
  vector<int> *totalFrame     = new vector<int>;
  vector<int> *nucData        = new vector<int>;
  vector<int> *regionsCovered = new vector<int>;

  for(f=0;f<12;f++) totalFrame->push_back(0);
  for(f=0;f<20;f++)  nucData->push_back(0);
  for(f=1;f<=m_seq->get_region_num();f++) {
    if (m_seq->get_region(f)->type==REGION_CEXON) totalExons++;
    regionsCovered->push_back(0);
  }
  regionsCovered->push_back(0);
  int judgeOnlyDP = reg->lookupVal("judgeOnlyDP").contains("yes",0);

  int plength   = parse->size();
  int *pregions = new int[plength]; arrayZero(pregions , plength);
  int *prstarts = new int[plength]; arrayZero(prstarts , plength);
  int *prstops  = new int[plength]; arrayZero(prstops  , plength);
  int *prframes = new int[plength]; arrayZero(prframes , plength);

  for (i=0; i<plength; ++i) {
    pregions[i] = (*parse)[plength-1-i];
    prstarts[i] = scorem->row(pregions[i]);
    prstops[i]  = scorem->col(pregions[i]);

    prframes[i] =  ( i==0 )  ?  prstarts[0] % 3  :  ( 3+ prframes[i-1] + prstarts[i] - prstops[i-1] - 1 ) %3;
  }

  for(int x=0; x<plength; ++x) {
    long int start   = prstarts[x];
    long int stop    = prstops[x];
    int myFrame = prframes[x];

    cout << scorem->get(pregions[x]) << " <- " << myFrame << "/";
    int framesPossible=stopMatrix->get(pregions[x]);

    if ((x==0)         && (endpointType[start]&ATG))        framesPossible |= (1 << ((start+1)%3)) | (1 << ((start+2)%3));
    if ((x==plength-1) && (endpointType[start]&STOP_CODON)) framesPossible |= (1 << (stop%3))      | (1 << ((stop-1)%3));

    int numFrames = 0;
    if ((framesPossible&1)!=1) { cout << "0"; numFrames++; }
    if ((framesPossible&2)!=2) { cout << "1"; numFrames++; }
    if ((framesPossible&4)!=4) { cout << "2"; numFrames++; }
    cout << " (" << start << "," << stop << ")";
    int overlap = 0;
    for(f=1;f<=m_seq->get_region_num();f++) {
      Region *region=m_seq->get_region(f);
      if ((region->type==REGION_CEXON)&&((!judgeOnlyDP)||
	 ((judgeOnlyDP)&&((endpointType[region->start]&1)==1)&&
	  ((endpointType[region->stop]&2)==2)))) {
	long int start2 = region->start;
	long int stop2  = region->stop;
	if (MAX(start,start2)<MIN(stop,stop2)) {
	  totalOverlap++;
	  (*regionsCovered)[f]=1;
	  if (overlap) cout << "\t \t \t";
	  double firstOverlap  = (((double)(MIN(stop,stop2)-MAX(start,start2)))/((double)(stop-start)))*100;
	  double secondOverlap = ((double)(MIN(stop,stop2)-MAX(start,start2)))/((double)(stop2-start2))*100; 
	  int    frame         = (3-region->frame+region->start)%3;

	  cout << "-(" << start2 << "," << stop2 << ") (" << firstOverlap << "% " << secondOverlap << "%)" << " " << frame;

	  if ((firstOverlap==100)&&(secondOverlap==100)) {
	    totalPerfect++;
	    (*totalFrame)[2*numFrames+5]++;
	    if (frame==myFrame) (*totalFrame)[2*numFrames+4]++;
	  }
	  (*totalFrame)[2*numFrames-1]++;
	  if (frame==myFrame) {
	    (*totalFrame)[2*numFrames-2]++;
	    cout << " MATCH! ";
	    totalMatch++;
	  }
	  cout << endl; 
	  overlap=1;
       	}
      }
    }
    if (!overlap) {
      totalNonOverlap++;
      cout << endl;
    }
  }

  cout << "Total overlaps: \t" << totalOverlap << endl;
  cout << "Total frame matches: \t" << totalMatch << endl;
  cout << "Frames correct (overlaps): ";

  for(f=0;f<6;f+=2) {
    cout << "(" << (*totalFrame)[f] << "/" << (*totalFrame)[f+1] << "=";
    cout << (((double)((*totalFrame)[f]))/((double)((*totalFrame)[f+1]))*100) << "%) ";
  }
  cout << ((double)((*totalFrame)[0]+(*totalFrame)[2]+(*totalFrame)[4]))/
    ((double)((*totalFrame)[1]+(*totalFrame)[3]+(*totalFrame)[5]))*100 << "% vs. ";
  cout << ((double)((*totalFrame)[1]+(((double)(*totalFrame)[3])/2)+(((double)(*totalFrame)[5])/3)))/
    ((double)((*totalFrame)[1]+(*totalFrame)[3]+(*totalFrame)[5]))*100 << "%" << endl;
  cout << "Frames correct (perfect): ";
  for(f=6;f<12;f+=2) {
    cout << "(" << (*totalFrame)[f] << "/" << (*totalFrame)[f+1] << "=";
    cout << (((double)((*totalFrame)[f]))/((double)((*totalFrame)[f+1]))*100) << "%) ";
  }
  cout << ((double)((*totalFrame)[6]+(*totalFrame)[8]+(*totalFrame)[10]))/
    ((double)((*totalFrame)[7]+(*totalFrame)[9]+(*totalFrame)[11]))*100 << "% vs. ";
  cout << ((double)((*totalFrame)[7]+(((double)(*totalFrame)[9])/2)+(((double)(*totalFrame)[11])/3)))/
    ((double)((*totalFrame)[7]+(*totalFrame)[9]+(*totalFrame)[11]))*100 << "%" << endl;
  cout << "Total perfect exons: \t" << totalPerfect << endl;
  cout << "Total false positives: \t" << totalNonOverlap << endl;
  cout << endl;
  cout << "Misses: " << endl;
  
  int totalMiss=0;
  if (!(judgeOnlyDP&&(plength==0))) 
    for(f=1;f<=m_seq->get_region_num();f++) 
      if (!(*regionsCovered)[f]) {
	Region *region=m_seq->get_region(f);
	if ((region->type==REGION_CEXON)&&((!judgeOnlyDP)||
	   ((judgeOnlyDP)&&((endpointType[region->start]&1)==1)&&
	    ((endpointType[region->stop]&2)==2)))) {
	  totalMiss++;
	  cout << "(" << region->start << "," << region->stop << ") ";
	}
      }

  cout << endl;
  String actualSeq, predictedSeq;
  for(f=plength-1;f>=0;f--) {
    int myFrameFplus1, myFrameF = prframes[plength-1-f];
    if (f<plength-1) myFrameFplus1 = prframes[plength-1-f-1]; 
    
    if (f < plength-1) {
      for(i=0;i<((myFrameFplus1 - myFrameF +framem->row( pregions[plength-1-f] ) - 
		  framem->col(pregions[plength-1-(f+1)]) + 5)%3);i++) predictedSeq+='n';
      predictedSeq+="   ";
    } 
    else {
      for (i=0;i<((myFrameF-((framem->row((*parse)[f]))%3)+3)%3);i++) 
	predictedSeq+='n';
    }
    for(i=framem->row((*parse)[f]);i<=framem->col((*parse)[f]);i++) 
      predictedSeq+=nucl2char(m_seq->get(i));
  }
  
  for(f=1;f<=seql;f++) {
    int overlapFlag=0;
    Region *region=m_seq->nuc2reg(f);
    if (region->type==REGION_CEXON) actualSeq+=nucl2char(m_seq->get(f));
    else if ((f>1)&&(m_seq->nuc2reg(f-1)->type==REGION_CEXON)) actualSeq+="   ";
    for(int x=0; x<plength; x++) 
      if ((scorem->row((*parse)[x])<=f)&&
	  (f<=scorem->col((*parse)[x]))) {
	overlapFlag=1;
	if (region->type==REGION_CEXON) {
	  (*nucData)[4]++;

	  int myFrameX = prframes[plength-1-x];
	  if (((3-region->frame+region->start)%3) == myFrameX) (*nucData)[8]++;
	} 
	else (*nucData)[5]++;
	break;
      } 
    if (!overlapFlag) {
      if (region->type==REGION_CEXON) (*nucData)[7]++;
      else                            (*nucData)[6]++;
    }
  }

  (*nucData)[8] -= (((*nucData)[8])%3);
  (*nucData)[0]  = (*nucData)[4] + (*nucData)[5];
  (*nucData)[1]  = (*nucData)[6] + (*nucData)[7];
  (*nucData)[2]  = (*nucData)[4] + (*nucData)[7];
  (*nucData)[3]  = (*nucData)[5] + (*nucData)[6];

  cout << endl;
  cout << "Total not covered: \t" << totalMiss << endl;
  cout << "Total exons: " << totalExons << endl;
  cout << "Total predicted exons: " << totalPExons << endl;
  cout << endl;
  double GCTotal=(*GCInfo)[0]+(*GCInfo)[1]+(*GCInfo)[2]+(*GCInfo)[3];
  cout << "ACGT content: (" << ((*GCInfo)[0]/GCTotal)*100 << "% ";
  cout << ((*GCInfo)[1]/GCTotal)*100 << "% " << ((*GCInfo)[2]/GCTotal)*100 << "% ";
  cout << ((*GCInfo)[3]/GCTotal)*100 << "%)" << endl;
  GCTotal=(*GCInfo)[4]+(*GCInfo)[5]+(*GCInfo)[6]+(*GCInfo)[7];
  cout << "(n*ACGT)/l: (" << ((*GCInfo)[4]/GCTotal)*100 << "% ";
  cout << ((*GCInfo)[5]/GCTotal)*100 << "% " << ((*GCInfo)[6]/GCTotal)*100 << "% ";
  cout << ((*GCInfo)[7]/GCTotal)*100 << "%)" << endl;

  cout << endl;

  cout << "Accuracy per nucleotide: Sn=";
  cout << (((double)((*nucData)[4]))/((double)((*nucData)[2])));
  cout << " Sp=" << (((double)((*nucData)[4]))/((double)((*nucData)[0])));
  cout << " AC=" << ((((((double)((*nucData)[4]))/((double)((*nucData)[2])))+
		       (((double)((*nucData)[4]))/((double)((*nucData)[0])))+
		       (((double)((*nucData)[6]))/((double)((*nucData)[3])))+
		       (((double)((*nucData)[6]))/((double)((*nucData)[1]))))/2)-1) << endl;
  cout << "PP=" << ((*nucData)[0]) << " PN=" << ((*nucData)[1]);
  cout << " AP=" << ((*nucData)[2]) << " AN=" << ((*nucData)[3]);
  cout << " TP=" << ((*nucData)[4]) << " FP=" << ((*nucData)[5]);
  cout << " TN=" << ((*nucData)[6]) << " FN=" << ((*nucData)[7]) << endl;
  cout << endl;
  cout << "Accuracy per exon: Sn=" << (((double)totalPerfect)/((double)totalExons));
  cout << " Sp=" << (((double)totalPerfect)/((double)totalPExons));
  cout << " ME=" << (((double)totalMiss)/((double)totalExons));
  cout << " WE=" << (((double)totalNonOverlap)/((double)totalPExons)) << endl;
  cout << "TE=" << totalPerfect << " PE=" << totalPExons << " AE=" << totalExons << endl;
  cout << endl;
  cout << ((((double)((*nucData)[8]))/((double)((*nucData)[2])))*100);
  cout << "% protein correct (lower bound)." << endl;
  cout << endl;

  cout << "Segments removed: " << myData[SEG_REMOVED] << endl;
  (*nucData)[9+SEG_REMOVED] += (int)myData[SEG_REMOVED];
   
  cout << "Two frame exons: " << myData[TWO_FRAME_EXONS] << endl;
  (*nucData)[9+TWO_FRAME_EXONS] += (int)myData[TWO_FRAME_EXONS];
  
  if (reg->lookupVal("generateParse").contains("protein")) {
    cout << "Exon positions covered: " << myData[EXON_POS_COVERED] << endl;
    (*nucData)[9+EXON_POS_COVERED] += (int)myData[EXON_POS_COVERED];
    cout << "Exon positions not covered: " << myData[EXON_POS_UNCOV] << endl;
    (*nucData)[9+EXON_POS_UNCOV] += (int)myData[EXON_POS_UNCOV];

    cout << "Intron positions covered: " << myData[INTRON_POS_COVERED] << endl;
    (*nucData)[9+INTRON_POS_COVERED] += (int)myData[INTRON_POS_COVERED];
    cout << "Intron positions not covered: " << myData[INTRON_POS_UNCOV] << endl;
    (*nucData)[9+INTRON_POS_UNCOV] += (int)myData[INTRON_POS_UNCOV];
  }      

  if (VERBOSE)
    cout << "Actual sequence: "    << actualSeq                                 << endl << endl
	 << "Predicted sequence: " << predictedSeq                              << endl << endl
	 << "Actual protein: "     << m_seq->DNAToProtein(actualSeq.chars())    << endl << endl
	 << "Predicted protein: "  << m_seq->DNAToProtein(predictedSeq.chars()) << endl << endl;
  
  assert(((*nucData)[4]+(*nucData)[5]+(*nucData)[6]+(*nucData)[7])==seql);

  argTotalOverlap     += totalOverlap;
  argTotalNonOverlap  += totalNonOverlap;
  argTotalPerfect     += totalPerfect;
  argTotalMatch       += totalMatch;
  argTotalMiss        += totalMiss;
  argTotalExons       += totalExons;
  argTotalPExons      += totalPExons;

  for(f=0; f<12; f++) (*argTotalFrame)[f] += (*totalFrame)[f];
  for(f=0; f<8;  f++)     (*argGCInfo)[f] +=     (*GCInfo)[f];
  for(f=0; f<20;  f++)    (*argNucData)[f] +=    (*nucData)[f];

  delete[] pregions;   delete[] prframes;
  delete[] prstarts;   delete[] prstops;

  delete GCInfo;         delete nucData;
  delete regionsCovered; delete totalFrame;
}

double Modules::frameMethod(double *frame, Method FrameM, int framesPossible)
  // Requires: frame is a double array of appropriate size;
  //           For instance, PERK, SPERK will use frame[0,1,2] which are assumed
  //           to be frame score sums e[i][j]+ ...
  //           while MININDEX will use frame[3,4,5] which are assumed to be
  //           sums of inverses: 1/e[i][j] + ...
{
  int nposs[3], i, index;
  for(i=0; i<3; i++) nposs[i] = (framesPossible >> i) & 1;

  double scores[3], median;


  switch (FrameM) {
  // Score methods:
  case Max:        
    return MAX(    nposs[0]? -Infinity : frame[0], 
		   nposs[1]? -Infinity : frame[1], 
		   nposs[2]? -Infinity : frame[2]);
  case Min:        
    return (1/MIN( nposs[0]? Infinity : frame[3],
		   nposs[1]? Infinity : frame[4],
		   nposs[2]? Infinity : frame[5]));
  case Perk:      
    for(i=0;i<3;i++) {
      scores[i]=2*frame[i]-frame[(i+2)%3]-frame[(i+1)%3];
      assert(scores[i] < 100000);
    }
  case SPerk:      
    if (FrameM!=Perk) {
      median=MEDIAN(scores[0],scores[1],scores[2]);
      for(i=0;i<3;i++) scores[i]=frame[i]-median;
    }
    
    if (!nposs[MAXINDEX(    scores[0] , scores[1] , scores[2])])
      return    MAX(        scores[0] , scores[1] , scores[2]);
    if (!nposs[MEDIANINDEX( scores[0] , scores[1] , scores[2])])
      return    MEDIAN(     scores[0] , scores[1] , scores[2]);
    if (!nposs[MININDEX(    scores[0] , scores[1] , scores[2])])
      return    MIN(        scores[0] , scores[1] , scores[2]);
    return -Infinity;
    
  // Frame methods:
  case MinIndex:   
    return MININDEX( nposs[0]?  Infinity : frame[3], 
		     nposs[1]?  Infinity : frame[4], 
		     nposs[2]?  Infinity : frame[5] );     
  case MaxIndex:   
    return MAXINDEX( nposs[0]? -Infinity : frame[0], 
		     nposs[1]? -Infinity : frame[1], 
		     nposs[2]? -Infinity : frame[2] );
    
  case PerkIndex:  for(i=0;i<3;i++) scores[i]=2*frame[i]-frame[(i-1)%3]-frame[(i+1)%3];

  case SPerkIndex: if (FrameM!=PerkIndex) {
                     median=MEDIAN(scores[0],scores[1],scores[2]);
		     for(i=0;i<3;i++) scores[i]=frame[i]-median;
                   }

                   if ((!nposs[index=MAXINDEX(   scores[0],scores[1],scores[2])])||
		       (!nposs[index=MEDIANINDEX(scores[0],scores[1],scores[2])])||
		       (!nposs[index=MININDEX(   scores[0],scores[1],scores[2])]))
		     return index;
		   return 3;

  default:         err("UNDEFINED FRAME METHOD");
                   return Infinity;
  }
}


