// -*- c++ -*-

#ifndef __MATRIX_H__
#define __MATRIX_H__

#include <math.h>
#include <stdio.h>
#include <iostream.h>
#include <String.h>

#include "vector.h"

template <class T>
class FullMatrix {
  // Created by Wes Beebee
public:
  FullMatrix(int rows, int cols, T initVal) { 
    nrows=rows; 
    ncols=cols;

    data=new (T*)[nrows];
    for(int z=0;z<nrows;z++) data[z]=new T[ncols];

    fill(initVal);
  }

  FullMatrix(int rows=0, int cols=0) { 
    nrows=rows; 
    ncols=cols;

    data=new (T*)[nrows];
    for(int z=0;z<nrows;z++) data[z]=new T[ncols];
  }

  ~FullMatrix() {
    for(int z=0;z<nrows;z++) delete[] data[z];
    delete[] data;
  } 

  istream& operator>>(istream& in) {
    String inputLine;
    if (!readline(in,inputLine)) return in;
    nrows=atoi(inputLine.before(",").chars());
    ncols=atoi(inputLine.after(",").chars());
    char c;
    
    for(int y=0;(y<nrows)&&in.good();y++)
      for(int z=0;(z<ncols)&&in.good();z++) {
	while (((c=in.peek())==' ')||(c=='\t')||(c=='\n')) in.get();
	if (in.good()) in >> data[y][z];
      }
    return in;
  }

  ostream& operator<<(ostream& out) {
    out << nrows << "," << ncols << endl;
    
    for (int z=0;z<nrows;z++) {
      for (int y=0;y<ncols;y++) 
	out << data[z][y] << "\t";
      out << endl;
    }
    return out;
  }

  inline int rows() {return nrows;};
  inline int cols() {return ncols;};
  inline T get(int row, int col) { 
    assert((row<nrows)&&(row>=0));
    assert((col<ncols)&&(col>=0));
    return data[row][col]; 
  }

  inline void set(int row, int col, T val) { 
    assert((row<nrows)&&(row>=0));
    assert((col<ncols)&&(col>=0));
    data[row][col]=val; 
  }

  void fill(T val, int endRows=-1, int endCols=-1, 
		   int beginRows=0, int beginCols=0) {
    for(int row=beginRows;row<((endRows==-1)?nrows:endRows);row++) 
      for(int col=beginCols;col<((endCols==-1)?ncols:endCols);col++)
	set(row,col,val);
  }

private:
  int nrows, ncols;
  T** data;
};


template <class T>
class SparseMatrix {
  // Overview: A sparse matrix only stores the locations and values of non-zero
  //           elements.  Thus, if you only have a few non-zero elements, a 
  //           sparse matrix is far more efficient to use.
  //
  //           You can also iterate through the non-zero values in a sparse 
  //           matrix.

  // Rep. Invariant: All entries in m_vdValue are non-zero.

  // Abstraction Function: The matrix A such that: 
  //                       forall i {(m_vdRow[i],m_vdCol[i])->m_vdVal[i])}
  //                       represents the location and value corresponding to 
  //                       all non-zero elements in A.
  //
  // Note: This sparse matrix is optimized for REALLY sparse matrices.
  //       The implementation is O(n) for most functions (instead of O(log n)),
  //       but has a relatively small runtime for small n.
  //
  //       An O(log n) implementation with more validity checks on input is
  //       recommended for larger "sparse" matrices.
  //
  // Created by Wes Beebee
public:
// Constructor

  SparseMatrix(int nRows, int nCols, T initVal)
    // Requires: "nRows">0 and "nCols">0 
    // Effects: constructs a sparse matrix with "nRows" rows and "nCols"
    //          columns, initialized to a value "initVal".
  {
    m_nRows=nRows; m_nCols=nCols; m_initVal=initVal;
  }

  ~SparseMatrix(){}

// Overloaded operators
  /*  SparseMatrix& operator = (const SparseMatrix& m)
    // Requires: "m" points to a valid matrix.
    // Modifies: "this" 
    // Effects: sets the value of "this" equal to the value of "m".
  {
    m_nRows=m.m_nRows; m_nCols=m.m_nCols; m_initVal=m.m_initVal;
    m_vnRow=m.m_vnRow; m_vnCol=m.m_vnCol; m_vVal=m.m_vVal;
    return *this;
  }*/

  /*  istream& operator>>(istream& in) 
  // Requires: input on stream "in" 
  // Modifies: "in", "m"
  // Effects: Stores values "row","col","val" into "m" while input remains
  //          valid.
  {
    vector<T> v; 
    vector<int> vn;
    m_vnRow=vn; m_vnCol=vn; m_vVal=v;  // initialize vectors
    assert(m_vnRow.length == 0);
    assert(m_vnCol.length == 0);
    
    int row, col;
    T val;
    while (cin >> row >> col >> val) addEntry(row,col,val);
    return in;
  }*/

  ostream& operator<<(ostream& out) 
  // Requires: Valid output stream "cout"
  // Modifies: "cout"
  // Effects: Outputs values for "row","col", and "val" 
  //          from "m" to "cout". 
  {
    for (int i=0;i<sizeNonZero();i++) 
      out << row(i) << " " << col(i) << " " << get(i) << endl;
    return out;
  }
  
  // Effects: Returns number of rows allocated for SparseMatrix.
  inline int rows() {return m_nRows;};
  inline int rows() const {return m_nRows;};

  // Effects: Returns number of columns allocated for SparseMatrix.
  inline int cols() {return m_nCols;};
  inline int cols() const {return m_nCols;};

  void set(int nRow, int nCol, T val) 
    // Modifies: "this"
    // Effects: Stores the value "val" in the location ("nRow","nCol").
    //          ASSERTs if input out of range.
  {
    int i=toNonZero(nRow, nCol);
    if (val==m_initVal) {
      if (i!=-1) removeEntry(i);
    } else {
      if (i==-1) addEntry(nRow,nCol,val);
      else { m_vnRow[i]=nRow; m_vnCol[i]=nCol; m_vVal[i]=val; }
    }
  }

  T get(int nRow, int nCol) 
    // Effects: Returns the value stored in the location ("nRow","nCol").
    //          ASSERTs if input out of range.
  {
    int i=toNonZero(nRow, nCol);
    if (i==-1) return m_initVal; else return get(i);
  }

  // Requires: The following three require a valid index "nNonZero"
  //           to the nth non-zero element in the matrix.
  inline T get(int nNonZero) { return m_vVal[nNonZero]; };
  // Effects: Returns the value of the nth non-zero element of the
  //          matrix.  (elements indexed in any order.)

  inline int row(int nNonZero) { return m_vnRow[nNonZero]; };
  // Effects: Returns the row of the nth non-zero element.

  inline int col(int nNonZero) { return m_vnCol[nNonZero]; };
  // Effects: Returns the column of the nth non-zero element.

  inline int sizeNonZero() { return m_vnRow.size(); };
  // Effects: Returns the number of non-zero elements in the matrix.

  int toNonZero(int nRow, int nCol) 
    // Effects: Returns the non-zero index of the element 
    //          at ("nRow","nCol")
  {
    int i=sizeNonZero()-1;
    assert((nRow<m_nRows)&&(nCol<m_nCols));
    while ((i>=0)&&((row(i)!=nRow)||(col(i)!=nCol))) i--;
    return i;
  }

  void addEntry(int nRow, int nCol, T val) 
    // Requires: (nRow,nCol) not already in matrix.
    // Effects: Quick theta(1) add to matrix.
  {
    if (val!=m_initVal) {
      m_vnRow.push_back(nRow); 
      m_vnCol.push_back(nCol); 
      m_vVal.push_back(val);
    }
  }

  void clear() 
    // Effects: Fills the matrix with the initial value.
  {
    m_vnRow.clear();
    m_vnCol.clear();
    m_vVal.clear();
  }

private:
  void removeEntry(int i) {
    m_vnRow.erase(i); 
    m_vnCol.erase(i); 
    m_vVal.erase(i);
  }

  int m_nRows;
  int m_nCols;
  vector<int> m_vnRow, m_vnCol; 
  vector<T> m_vVal;
  T m_initVal;
};



template <class T>
class MySparseMatrix {
public:
  MySparseMatrix(int nRows, int nCols, T initVal) {
    m_nRows=nRows; m_nCols=nCols; m_initVal=initVal;
  }

  ~MySparseMatrix(){}
  ostream& operator<<(ostream& out) {
    for (int i=0;i<sizeNonZero();i++) 
      out << row(i) << " " << col(i) << " " << get(i) << endl;
    return out;
  }
  
  inline int rows() {return m_nRows;};
  inline int rows() const {return m_nRows;};

  inline int cols() {return m_nCols;};
  inline int cols() const {return m_nCols;};

  void set(int nRow, int nCol, T val) {
    int i=toNonZero(nRow, nCol);
    if (val==m_initVal) {
      if (i!=-1) removeEntry(i);
    } else {
      if (i==-1) addEntry(nRow,nCol,val);
      else { m_vnRow[i]=nRow; m_vnCol[i]=nCol; m_vVal[i]=val; }
    }
  }

  T get(int nRow, int nCol) {
    int i=toNonZero(nRow, nCol);
    if (i==-1) return m_initVal; else return get(i);
  }

  inline T get(int nNonZero) { return m_vVal[nNonZero]; };
  inline int row(int nNonZero) { return m_vnRow[nNonZero]; };
  inline int col(int nNonZero) { return m_vnCol[nNonZero]; };
  inline int sizeNonZero() { return m_vnRow.size(); };

  int toNonZero(int nRow, int nCol) {
    int i=sizeNonZero()-1;
    assert((nRow<m_nRows)&&(nCol<m_nCols));
    while ((i>=0)&&((row(i)!=nRow)||(col(i)!=nCol))) i--;
    return i;
  }

  void addEntry(int nRow, int nCol, T val) {
    if (val!=m_initVal) {
      m_vnRow.push_back(nRow); 
      m_vnCol.push_back(nCol); 
      m_vVal.push_back(val);
    }
  }

  void clear() {
    m_vnRow.clear();
    m_vnCol.clear();
    m_vVal.clear();
  }

private:
  void removeEntry(int i) {
    m_vnRow.erase(i); 
    m_vnCol.erase(i); 
    m_vVal.erase(i);
  }

  int m_nRows,m_nCols;
  
  vector<int> m_vnRow, m_vnCol; 
  vector<T> m_vVal;
  T m_initVal;
};

template <class T>
class HashSparseMatrix {
public:
  HashSparseMatrix(int nRows, int nCols, T initVal) {
    m_nRows=nRows; 
    m_nCols=nCols; 
    m_initVal=initVal;
    m_vnVals=new vector<T>[nRows];
    m_vnCols=new vector<int>[nRows];
  }

  ~HashSparseMatrix() {
    delete[] m_vnVals;
    delete[] m_vnCols;
  }

  void clear() {
    int i;
    for(i=0;i<m_nRows;i++) {
      (m_vnVals[i]).clear();
      (m_vnCols[i]).clear();
    }
  }

  inline int rows() {return m_nRows;};
  inline int rows() const {return m_nRows;};

  inline int cols() {return m_nCols;};
  inline int cols() const {return m_nCols;};
  
  T get(int nRow, int nCol) {
    int i, k = (m_vnCols[nRow]).size();
    for(i=0; i<k; i++) 
      if (m_vnCols[nRow][i]==nCol) 
	return m_vnVals[nRow][i];
    return m_initVal;
  }

  void set(int nRow, int nCol, T val) {
    assert((0<=nRow)&&(nRow<m_nRows)&&(0<=nCol)&&(nCol<m_nCols));
    int i, k=(m_vnCols[nRow]).size();
    for(i=0; i<k; i++)
      if (m_vnCols[nRow][i]==nCol) {
	if (val==m_initVal) {
	  m_vnVals[nRow].erase(i);
	  m_vnCols[nRow].erase(i);
	} else m_vnVals[nRow][i]=val;
	return;
      }

    if (val!=m_initVal) {
      m_vnCols[nRow].push_back(nCol);
      m_vnVals[nRow].push_back(val);
    }
  }

  void addEntry(int nRow, int nCol, T val) {
    assert((0<=nRow)&&(nRow<m_nRows)&&(0<=nCol)&&(nCol<m_nCols));
    if (val!=m_initVal) {
      m_vnCols[nRow].push_back(nCol);
      m_vnVals[nRow].push_back(val);
    }
  }

private:
  int m_nRows,m_nCols;
  vector<T>* m_vnVals;
  vector<int>* m_vnCols;
  T m_initVal;
};

#endif // __MATRIX_H__

// End of header file




















