#include "common.h"
// Modified for new seq specs;

void warn(char *buff) {
  cout << "WARNING: " << buff << endl;
}
 
void err(char *buff) {
  cout << "ERROR: " << buff << endl;
  exit(0);
}
void verb(char *buff) {
  if (VERBOSE) cout << buff << endl;
}

void cond(char *buff, int condition) {
  if (condition) cout << *buff << endl;
}
void verbC(char *buff, int condition) {
  if (VERBOSE && condition) cout << *buff << endl;
}



Nucleotide char2nucl(char ch) {
  switch (ch) {
  case 'a':
  case 'A':
    return BASE_A;
  case 'c':
  case 'C':
    return BASE_C;
  case 'g':
  case 'G':
    return BASE_G;
  case 't':
  case 'T':
    return BASE_T;
  default:
    return BASE_UNKNOWN;
  }
}


char nucl2char(Nucleotide base) {
  switch (base) {
    case BASE_A:
      return 'a';
    case BASE_C:
      return 'c';
    case BASE_G:
      return 'g';
    case BASE_T:
      return 't';
  default:
    return 'n';
  }
}



int power(int n, int k) {
  if (k==0) return 1;
  else return n*power(n,k-1);
}

int patt_to_index(Sequence &seq, int pos, int n) {
  if (n == 1) return (int)(seq.get(pos));
  return(power(4,n-1)*((int)(seq.get(pos))) + patt_to_index(seq, pos+1, n-1));
}

void index_to_patt(int index, char *patt, int l) {
  assert(l>0); assert(index>=0); assert(patt);
  
  if ((index % 4) == 0) patt[l-1] = 'a';
  else if ((index % 4) == 1) patt[l-1] = 'c';
  else if ((index % 4) == 2) patt[l-1] = 'g';
  else if ((index % 4) == 3) patt[l-1] = 't';

  if (l>1) index_to_patt(index/4, patt, l-1);
}

int Pattern::index() { // Not efficient but that is not important for the Pattern class;
  if (!size) {
    warn("index requested for empty pattern");
    return -1;
  }
  int idx=0,i=0;
  
  for (i=0; i<size; ++i) {
    assert((int)patt[i]>=0 && (int)patt[i]<4);
    idx += power(4,size-(i+1)) * (int) patt[i];
  }
  return idx;
}

Pattern::Pattern() { 
  size = 0;
  patt = NULL;
}



Pattern::~Pattern() {
  if (patt) delete[] patt;
  patt = NULL;
}

Pattern::Pattern(const char *buff, int sz) {
  size = sz;
  patt = new Nucleotide[sz];
  
  for (int i=0; i<sz; ++i)      
    patt[i] = char2nucl(buff[i]);
}    


bool Pattern::seq_exhibits(Sequence &seq, int pos, int n) {
//  assert(patt && n <= size);
//  assert(seq.get_length() >= pos+n-1 && pos >= 0);
  
  for (int i=0; i<n; ++i)
    if (seq.get(i+pos) != patt[i]) return FALSE;
  return TRUE;
}

ostream& operator<<(ostream& cout, Pattern& p) {
  for (int i=0; i<p.size; ++i)
    cout << nucl2char(p.patt[i]);
  return cout;
}

double align(int *first, int *second, int fromfirst, int tofirst, int fromsecond, int tosecond, 
	     double match, double mismatch, double gap, int *imagefirst, int *imagesecond){
  
  // Performs a global alignment on two sequences. M contains the scores, P contains
  // the pointers. The encoding is 0=up, 1=left, 2=upleft
  
  int i,j;
  double up, left, upleft;
  double score=0;
  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;
  double **M=new (double*)[lengthfirst+2];
  double **P=new (double*)[lengthfirst+2];
  double **Bonus=new (double*)[lengthfirst+2];
  for (i=0; i<lengthfirst+2; ++i){
    M[i]=new double[lengthsecond+2];
    P[i]=new double[lengthsecond+2];
    Bonus[i]=new double[lengthsecond+2];
  }

  // Boundary Conditions:
  for (i=0; i<lengthfirst+1; ++i){
    M[i][0]=gap*i;
    P[i][0]=0;
  }
  for (j=0; j<lengthsecond+1; ++j){
    M[0][j]=gap*j;                 // M[0][j]=gap*i -change-;
    P[0][j]=1;
  }
  for (i=1; i<lengthfirst+1; ++i){
    for (j=1; j<lengthsecond+1; ++j){
      up=M[i-1][j]+gap;
      left=M[i][j-1]+gap;
      upleft=M[i-1][j-1]+( (first[fromfirst+i-1]==second[fromsecond+j-1]) ? match : mismatch);
      M[i][j]=MAX(up,left,upleft);
      if (up>=left && up>=upleft)
	P[i][j]=0;
      if (left>=up && left>=upleft)
	P[i][j]=1;
      if (upleft>=up && upleft>=left)
	P[i][j]=2;
    }
  }
  // find optimal alignment and fill in the arrays imagefirst and imagesecond
  i=lengthfirst;
  j=lengthsecond;
  score=M[i][j];

  while (i>0 || j>0){
    assert(P[i][j] == 0 || P[i][j] == 1 || P[i][j] == 2);

    if (P[i][j]==0){
      imagefirst[i-1]=-1;
      i--;
     continue;
    }
    if (P[i][j]==1){
      imagesecond[j-1]=-1;
      j--;
      continue;
    }
    if (P[i][j]==2){
      imagefirst[i-1]=j-1;
      imagesecond[j-1]=i-1;
      i--;
      j--;
      continue;
    }
  }

  assert(imagefirst[0]             >= 0 || imagesecond[0]              >= 0);
  assert(imagefirst[lengthfirst-1] >= 0 || imagesecond[lengthsecond-1] >= 0);

  int lastImageFirst = -1;
  for (i=0; i<lengthfirst; ++i) {
    assert(imagefirst[i] == -1 || imagefirst[i] > lastImageFirst);
    
    if (imagefirst[i] >=0) {
      lastImageFirst = imagefirst[i];
      assert(imagesecond[imagefirst[i]] == i);
      assert(imagefirst[i] < lengthsecond);
    }
  }

  // cleanup
  for (i=0; i<lengthfirst+2; ++i){
    delete[] M[i];
    delete[] P[i];
    delete[] Bonus[i];
  }
  delete[] P;
  delete[] M;
  delete[] Bonus;

  return score;
}

int find_match(FilterSequence *seqm, FilterSequence *seqh, int from, int to, int pos){
  
  assert(from<to);
  
  int bestpos=0, currmatch=0, bestmatch=0;
  for (int place=from; place <=to; ++place){
    currmatch=0;
    for (int matloop=max(place-4,from); matloop <=min(place+4,to); ++matloop){
      if (seqm->get(matloop) == seqh->get(pos+4-(min(place+4,to)-matloop)))
	currmatch++;
    }
    if (currmatch>bestmatch ||
	(currmatch==bestmatch && 
	 ABSDIFF((double)place,  ((double)to+(double)from)/2) < 
	 ABSDIFF((double)bestpos,((double)to+(double)from)/2))) {
      bestmatch=currmatch;
      bestpos=place;
    }
  }
  
  return bestpos;
  
} 

void mark_sequence(FilterSequence &seq, int l, int *marks) {
  int i=0,j=0,k=0;
  marks[0]=-1;
  for (i=1; i<=seq.get_length(); ++i)
    marks[i]=GARB;

  for (i=1; i<=seq.get_region_num(); ++i) {
    Region *r = seq.get_region(i);
    switch (r->type) 
      {
      case REGION_INTRON: {
	if (i<seq.get_region_num()) {
	  Region *rnext = seq.get_region(i+1);
	  if (rnext->type == REGION_CEXON || rnext->type == REGION_NCEXON)
	    for (k=r->stop-l+2; k>=r->start && k<=r->stop; ++k)
	      marks[k] = ITOE;
	}
	for (j=r->start; j<=r->stop-l+1; ++j)
	  marks[j]=INTR;
	break;
      }
      case REGION_CEXON: {
	int offset;
	switch (r->frame) {
	case FRAME_0: offset=0; break;
	case FRAME_1: offset=1; break;
	case FRAME_2: offset=2; break;
	default: offset=0;
	}
	for (j=r->start; j<=r->stop-l+1; ++j)
	  marks[j] = (j-r->start+offset)%3;
	if (i<seq.get_region_num()) {
	  Region *rnext = seq.get_region(i+1);
	  if (rnext->type == REGION_INTRON)
	    for (k=r->stop-l+2; k>=r->start && k<=r->stop; ++k)
	      marks[k] = ETOI;
	  else if (rnext->type == REGION_NCEXON)
	    for (k=r->stop-l-1; k>=r->start && k<=r->stop; ++k)
	      // We want all the tuples intersecting the stop codon;
	      //   to be considered being in the CTONC boundary region;
	      marks[k] = CTONC;
	}
	else {
	  for (k=r->stop-l-1; k>=r->start && k<=r->stop; ++k)
	    marks[k] = CTONC;
	  warn("Sequence ends with a coding exon.");
	}
	break;
      }
      case REGION_NCEXON: {
	if (i<seq.get_region_num()) {
	  Region *rnext = seq.get_region(i+1);
	  if (rnext->type == REGION_INTRON)
	    for (k=r->stop-l+2; k>=r->start && k<=r->stop; ++k)
	      marks[k] = ETOI;
	  else if (rnext->type == REGION_CEXON)
	    for (k=r->stop-l+2; k>=r->start && k<=r->stop; ++k)
	      marks[k] = CTONC;
	}
	for (j=r->start; j<=r->stop-l+1; ++j)
	  marks[j] = NCEX;
	break;
      }
      default: break;
      }
  }
  if (GARBAGE_IGNORE) {
    for (i=1; i<=seq.get_length(); ++i)
      if ((int)seq.get(i) < 0 || (int) seq.get(i) > 3)
	for (j=i-l+1; j<=i; ++j)
	  if (j>=0) marks[j] = GARB;
  }
}

int Partition(int *vals, int *indices, int p, int r) {
  int x = vals[p];
  int i = p-1, j=r+1;
  
  while (1) {
    do { j--; } while (vals[j] > x);
    do { i++; } while (vals[i] < x);
    if (i<j) {
      int tempval = vals[i];
      vals[i]     = vals[j];
      vals[j]     = tempval;
      int tempidx = indices[i];
      indices[i]  = indices[j];
      indices[j]  = tempidx;
    }
    else {
      assert(j >= p);
      for (int k=p; k<=j; ++k) {
	if (vals[k] > x) {
	  cout << "x=" << x << "   j=" << j << endl;
	  for (int k2=p; k2<=r; ++k2) {
	    if (!(k2 % 50) || k2==j) cout << k2 << ":"; 
	    cout << vals[k2] << " ";
	    if (!(k2%50)) cout << endl;
	  }
	}
	assert(vals[k] <= x);
      }
      assert(j < r);
      return j;
    }
  }
}

void Quicksort(int *vals, int *indices, int p, int r) {
  if (p<r) {
    int q;
    q = Partition(vals, indices, p,r);
    Quicksort(vals, indices, p, q);
    Quicksort(vals, indices, q+1, r);
  }
}



void orWrapper(long unsigned int *hits, vector<unsigned long int> pos, vector<unsigned long int> length){
  
  for (int j=0; j<pos.size(); j++){
    for (int k=pos[j]; k<pos[j]+length[j]; k++)
      hits[k]++;
  }
}

int fixUnknownNuc(FilterSequence* seq) {
  int fixed = 0;
  for(int i=0;i<seq->get_length();i++) 
    if ((*seq)[i]==BASE_UNKNOWN) {
      (*seq)[i]=BASE_C;
      fixed++;
    }
  return fixed;
}

