// This file is for commonly used constants, includes, and basic functions.
// Modified for new seq specs;

#ifndef COMMON_H
#define COMMON_H

#include "fseq.h"
#include <fstream.h>
#include <math.h>
#include <assert.h>
 
#include <stdlib.h>
#include <minmax.h>
#include "strstream.h"
#include "vector.h"
#include "ivector.h"
#define VERBOSE 0
#define ALIGNVERBOSE 0
// Global constants:
#define FALSE      false
#define TRUE       true
#define MAXSEQLEN  400000

#define SMALL_VALUE 0.0000000000001  // Twelve 0s and then 1;

#define GARBAGE_IGNORE 1

#ifdef  DEBUG
#include "dmalloc.h"
#endif

#define Inf Infinity

int isFinite(double x);

int power(int n, int k);
double power(double n, int k);

const double Infinity = HUGE_VAL;

Nucleotide char2nucl(char ch);

char nucl2char(Nucleotide base);

int patt_to_index(Sequence &seq, int pos, int n);
// Converts pattern of length n starting at pos in seq;
//   into an integer;

void index_to_patt(int index, char *patt, int l);
// Inverse of patt_to_index;
// Stores the result in string patt;

inline double abs(double d) { return d<0 ? -d : d; }


class Pattern {
// This is the definition of class Pattern;
// This is a nice way of finding patterns of arbitrary size;
//   in a sequence, but there are more efficient ways to search;
//   for many patterns simultaneously;

 public:
  int size;
  Nucleotide *patt;

  Pattern();
  ~Pattern();
  Pattern(const char *buff, int sz);

  int index();
  bool seq_exhibits(Sequence &seq ,int pos, int n);
  friend ostream& operator<<(ostream&, Pattern&);
};

// End of definition of class Pattern;;

enum TupleRegionType {FR0=0, FR1=1, FR2=2, INTR=3, NCEX=4,
		      ITOE=5, ETOI=6, CTONC=7, NCTOC=8, GARB=9};
// CAUTION: The values in the above enumerator are NOT TO BE CHANGED !!!;

int find_match(FilterSequence *seqh, FilterSequence *seqm, int from, int to, int pos);

// The function find_match takes as input two sequences, a position in the
// first, and returns the best match to this in the second sequence (between
// positions from and to). This is used for the mouse homology

void mark_sequence(FilterSequence &seq, int l, int *marks);

// This function marks a sequence as follows:
// 'marks' should be an integer array at least the length of the sequence;
// The marks are the above defined values, from 0 (FR0) to 7 (GARB);
// and their meanings are the intuitive ones. One important thing to;
// keep in mind is that if position i has mark FR0 then frame 0 has (i,i+1,i+2);
// as a codon;



// Templated generic functions:

template <class T> inline T MIN(const T& t1,const T& t2)                          
{ return (t1<t2)?t1:t2;             }

template <class T> inline T MIN(const T& t1,const T& t2,const T& t3)              
{ return MIN(t1,MIN(t2,t3));        }

template <class T> inline T MIN(const T& t1,const T& t2,const T& t3, const T& t4) 
{ return MIN(MIN(t1,t2),MIN(t3,t4));}

template <class T> inline T MAX(const T& t1,const T& t2)                          
{ return (t1>t2)?t1:t2;             }

template <class T> inline T MAX(const T& t1,const T& t2,const T& t3)              
{ return (t1>t2) ? ((t1>t3) ? t1 : t3) : ((t2>t3) ? t2 : t3); }

template <class T> inline T MAX(const T& t1,const T& t2,const T& t3,const T& t4)  
{ return MAX(MAX(t1,t2),MAX(t3,t4));}

template <class T> inline T MEAN(const T& t1,const T& t2)                         
{ return (t1+t2)/2;                 }

template <class T> inline T MEAN(const T& t1,const T& t2, const T& t3)            
{ return (t1+t2+t3)/3;              }

template <class T> inline T MEDIAN(const T& t1,const T& t2, const T& t3) {
  if ((t1 >= t2 || t3 >= t2) && (t1 <= t2 || t3 <= t2)) return t2;
  if ((t2 >= t3 || t1 >= t3) && (t2 <= t3 || t1 <= t3)) return t3;
  if ((t3 >= t1 || t2 >= t1) && (t3 <= t1 || t2 <= t1)) return t1;
  assert(0==1);
}


template <class T> inline T ABSDIFF(const T& t1, const T& t2) {
  if (t1>t2) return t1-t2;
  else return t2-t1;
}

template <class T> inline T SQR(const T& t1) {return (t1*t1);}
template <class T> inline T SQRT(const T& t1) {assert(t1>=0); return sqrt(t1);}
template <class T> inline T LOG(const T& t1) {return ((t1>0)?log(t1):-Infinity);}

template <class T> inline T SD(const T& t1, const T& t2, const T& t3) {
  return SQRT(SQR(t1-MEAN(t1,t2,t3))+SQR(t2-MEAN(t1,t2,t3))+SQR(t3-MEAN(t1,t2,t3)));
}

template <class T> inline T PERK(const T& t1, const T& t2, const T& t3) {
  return 3*MAX(t1,t2,t3)-t1-t2-t3;
}

template <class T> inline T SPERK(const T& t1, const T& t2, const T& t3) {
  return MAX(t1,t2,t3)-MEDIAN(t1,t2,t3);
}

template <class T> inline int MININDEX(const T& t1, const T& t2, const T& t3) {
  return ((t1==Infinity)&&(t2==Infinity)&&(t3==Infinity))?3:
    (((t1<=t2)&&(t1<=t3))?0:(((t2<=t1)&&(t2<=t3))?1:2));}

template <class T> inline int MAXINDEX(const T& t1, const T& t2, const T& t3) {
  return ((t1==Infinity)&&(t2==Infinity)&&(t3==Infinity))?3:
    (((t1>=t2)&&(t1>=t3))?0:(((t2>=t1)&&(t2>=t3))?1:2));}

template <class T> inline int MEDIANINDEX(const T& t1, const T& t2, const T& t3) {
  return ((t1==Infinity)&&(t2==Infinity)&&(t3==Infinity))?3:
    ((((t1>=t2)||(t3>=t2))&&((t1<=t2)||(t3<=t2)))?1:
    ((((t2>=t3)||(t1>=t3))&&((t2<=t3)||(t1<=t3)))?2:
    ((((t3>=t1)||(t2>=t1))&&((t3<=t1)||(t2<=t1)))?0:3)));
}

template <class T>
String toa(T i) {
  ostrstream tempStr;
  tempStr << i;
  String temp=tempStr.str();
  if (temp.length()>(unsigned int)tempStr.pcount()) 
    temp=temp.before(tempStr.pcount());
  return temp;
}

template <class T>
void froma(char* s, T& answer) {
  istrstream tempStr(s);
  tempStr >> answer;
}

void warn( char *buff);                // Warning;
void err(  char *buff);                // Error, stops the program;
void verb( char *buff);                // Output only if verbose;
void cond( char *buff, int condition); // Output only if condition;
void verbC(char *buff, int condition); // Output only if verbose and condition;


template <class T> inline void arrayInit(T val, T *arr, int sz) {
  int i;
  for (i=0; i<sz; arr[i++]=val);
}

void arrayZero(char          *arr, int sz);
void arrayZero(double        *arr, int sz);
void arrayZero(unsigned long *arr, int sz);
void arrayZero(int           *arr, int sz);

// Quicksort code follows;
int Partition(int *vals, int *indices, int p, int r);
void Quicksort(int *vals, int *indices, int p, int r);
// vals is an array of values, and indices is an array that;
//  should be initialized to [1 2 3 ... n];
// Then, vals will be modified to be the sorted values, and;
//  indices will be permuted exactly the same way;


// The protein wrappers follow:
void orWrapper(long unsigned int *hits, int *pos, int *lengths, int arrayl);

int fixUnknownNuc(FilterSequence* seq);
// Fixes the unknown nucleotides of the sequence: converts them to BASE_C;
// Returns number of nucleotides converted to BASE_C;

int cutUnknownNuc(FilterSequence *seq, char *ofname);

int isStopCodon(Nucleotide n1, Nucleotide n2, Nucleotide n3);


extern fstream GTfn;
extern fstream AGfn;

extern fstream GTeff;
extern fstream AGeff;

extern int GAGbadPyr;
extern int AGGoodRules[];
extern int GTGoodRules[];
extern fstream unknownExonRepeats;
extern fstream infoOut;

#endif













