#include "align.h"

double **currAlignPAM;

void outputPam(double **pamMat) {
  for (int i=0; i<22; ++i) {
    for (int j=0; j<22; ++j) cout << pamMat[i][j] << " ";
    cout << endl;
  }
}

void readCodonMatrix(double ***codonMat, ifstream &fin) {
  
  char buf[1000];
  int first, second, third, i;
  
  for (i=0; i<3; i++)
    fin.getline(buf,1000); // read in unnecessary headers
  
  char c1,c2,c3; double d;
  for (int line  = 0; line < 64; line++){
    
    if (line%4==0)
      fin.getline(buf,1000);
    first=((int)floor((line/16)))%16; second=((int)(floor((line/4))))%4; third=line%4; // calculate codon numbers
    first=(first+2)%4; second=(second+2)%4; third=(third+2)%4; // reindex to our system: A=0, C=1, G=2, T=3 
    if (first==0) first=3; else if (first==3) first=0;
    if (second==0) second=3; else if (second==3) second=0;
    if (third==0) third=3; else if (third==3) third=0;
    
    fin >> c1; fin >> c2; fin >> c3; // read name of amino acid
    fin >> c1; fin >> c2; fin >> c3; // read codon
    fin >> d; // read number of times codon occured
    fin >> codonMat[first][second][third]; // read in number of times codon occurs out of 1000
    codonMat[first][second][third]/=1000; // normalize matrix to be relative frequency
    fin >> d; // read fraction of times codon occurs for given amino acid
  }
}
 

void readPamMatrix(double **pamMat, ifstream &fin) {
  
  char buf[1000];
  
  do { 
    fin.getline(buf,1000);
  }
  while (buf[0] == '#');
  
  char c;
  fin >> c;
  assert(c=='A');

  for (int line = 0; line < 24; ++line)
    if (line > 19 && line != 23)       for (int col = 0; col < 25; ++col) fin >> buf;
    else if (line == 23) for (int col = 0; col < 22; ++col) fin >> pamMat[21][col];
    else
      for (int col = 0; col < 25; ++col) {
	if (col == 23)                                      fin >> pamMat[line][21];
	else if (col > 19)                                  fin >> buf;
	else                                                fin >> pamMat[line][col];
      }
  
  for (int i = 0; i<22; ++i) pamMat[i][20] = pamMat[20][i] = -Inf;
}


void normalizePam(double diagAvg, double **pamMat) {
  int i,j;
  double currSum=0;
  for (i=0; i<20; ++i) currSum += pamMat[i][i];
  double factor = diagAvg*20.0/currSum;
  // double factor = log(2)/2;

  for (i=0; i<22; ++i)
    for (j=0; j<22; ++j) pamMat[i][j] *= factor;
}


int breakptFind(int *htuples, int *hidx, int *himage, int hl,
		int *mtuples, int *midx, int *mimage, int ml, int tl,
		int *hbreaks, int *mbreaks) {
  // htuples, hidx, and himage are all 0-indexed;
  // hidx index a 1-indexed human sequence (all above same for mouse);
  // hbreaks and mbreaks are pre-allocated arrays of lengths hl, ml;
  //  that will contain the breakpts upon completion of the function call;
  // The number of breakpoints will be returned;

  int i=0,j=0;
  
  // Find matches;
  int *hmatched = new int[hl]; arrayZero(hmatched,hl);
  int *mmatched = new int[ml]; arrayZero(mmatched,ml);

  for (i=0; i<hl; ++i)
    if (himage[i] >=0)
      if (htuples[i] == mtuples[himage[i]]) {
	hmatched[i] = mmatched[himage[i]] = 1;
	assert(mimage[himage[i]] == i);
      }
  
  // Find consistent matches;
  int *hconsistent = new int[hl]; arrayInit(1, hconsistent, hl);
  int *mconsistent = new int[ml]; arrayInit(1, mconsistent, ml);

  for (i=0; i<hl; ++i) if (!hmatched[i]) hconsistent[i] = 0;
  for (i=0; i<ml; ++i) if (!mmatched[i]) mconsistent[i] = 0;

  for (i=1; i<hl; ++i)
    if (hmatched[i]) {
      j=i-1; 
      while (j>=0) {
	if (hmatched[j]) {
	  if (hidx[i] - hidx[j] >= tl) break;
	  if (midx[himage[i]] - midx[himage[j]] != hidx[i] - hidx[j])
	    hconsistent[i] = hconsistent[j] = mconsistent[himage[i]] = mconsistent[himage[j]] = 0;
	}
	j--;
      }
    }
  for (i=1; i<ml; ++i) 
    if (mmatched[i]) {
      j=i-1; 
      while (j>=0) {
	if (mmatched[j]) {
	  if (midx[i] - midx[j] >= tl) break;
	  if (midx[i] - midx[j] != hidx[mimage[i]] - hidx[mimage[j]])
	    mconsistent[i] = mconsistent[j] = hconsistent[mimage[i]] = hconsistent[mimage[j]] = 0;
	}
	j--;
      }
    }
  // Save results to hbreaks, mbreaks;  
  int breakCnt = 0;
  
  for (i=0; i<hl; ++i)
    if (hconsistent[i]) {
      assert(mconsistent[himage[i]]);
      hbreaks[breakCnt] = hidx[i];
      mbreaks[breakCnt] = midx[himage[i]];
      if (breakCnt) assert(mbreaks[breakCnt] > mbreaks[breakCnt-1]);
      if (breakCnt && (hbreaks[breakCnt] - hbreaks[breakCnt-1] < tl || mbreaks[breakCnt] - mbreaks[breakCnt-1] < tl))
	assert(hbreaks[breakCnt]-hbreaks[breakCnt-1] == mbreaks[breakCnt] - mbreaks[breakCnt-1]);
      
      breakCnt++;
    }
  
  
  delete[] hmatched; delete[] hconsistent;
  delete[] mmatched; delete[] mconsistent;
  
  return breakCnt;
}


int constrainedAlign(FilterSequence *hseq, FilterSequence *mseq,
		     int *hbreaks, int *mbreaks, int breakCnt, int tuplength,
		     double match, double mismatch, double gap, 
		     int *himage, int *mimage) {
  return constrainedAlign(hseq, mseq, hbreaks, mbreaks, breakCnt, tuplength, match, mismatch, gap, himage, mimage, 0, -10000);
}




int constrainedAlign(FilterSequence *hseq, FilterSequence *mseq,
		     int *hbreaks, int *mbreaks, int breakCnt, int tuplength,
		     double match, double mismatch, double gap, 
		     int *himage, int *mimage, int extnLength, int extnCutoff) {
  assert(breakCnt>0);

  int i=0, j=0, hseql = hseq->get_length(), mseql = mseq->get_length();
  
  arrayInit(-1,himage,hseql+1);
  arrayInit(-1,mimage,mseql+1);

  int *hseqint = new int[hseql+1]; seq2int(hseq,hseqint);
  int *mseqint = new int[mseql+1]; seq2int(mseq,mseqint);
  double localScore = 0;

  // First alignment;
  
  if (ALIGNVERBOSE) { for (i=0; i<breakCnt; ++i) cout << " ************* " << hbreaks[i] << " -> " << mbreaks[i] << endl; }
  
  int extended = 0;
  if (hbreaks[0] > 1 && mbreaks[0] > 1) {
    // extensions BEGIN;
    if (extnLength>0 && 
	hbreaks[0] > extnLength && 
	mbreaks[0] > extnLength) {
      localScore = gappedAlign(hseqint, mseqint, hbreaks[0]-extnLength, hbreaks[0]-1,
			       mbreaks[0]-extnLength, mbreaks[0]-1,
			       match, mismatch, 3*gap, gap, himage+hbreaks[0]-extnLength, mimage+mbreaks[0]-extnLength);
      if (localScore >= extnCutoff) {
	extended = 1;
	for (j=hbreaks[0]-extnLength; j<hbreaks[0]; ++j) if (himage[j]>=0) himage[j] += mbreaks[0]-extnLength;
	for (j=mbreaks[0]-extnLength; j<mbreaks[0]; ++j) if (mimage[j]>=0) mimage[j] += hbreaks[0]-extnLength;

	if (hbreaks[0] - extnLength > 1 && mbreaks[0] - extnLength > 1) {
	  gappedAlign(hseqint, mseqint, 1, hbreaks[0]-extnLength-1, 1, mbreaks[0]-extnLength-1, match, mismatch, 2*gap, gap, himage+1, mimage+1);
	  for (j=1; j<hbreaks[0]-extnLength; ++j) if (himage[j]>=0) himage[j]++;
	  for (j=1; j<mbreaks[0]-extnLength; ++j) if (mimage[j]>=0) mimage[j]++;
	}
      }
    }
    if (!extended) {
      for (j=1; j<hbreaks[0]; himage[j++]=-1);
      for (j=1; j<mbreaks[0]; mimage[j++]=-1);
      
      gappedAlign(hseqint, mseqint, 1, hbreaks[0]-1, 1, mbreaks[0]-1, match, mismatch, 2*gap, gap, himage+1, mimage+1);
      for (j=1; j<hbreaks[0]; ++j) if (himage[j]>=0) himage[j]++;
      for (j=1; j<mbreaks[0]; ++j) if (mimage[j]>=0) mimage[j]++;
    }
    
    for (j=1; j<hbreaks[0]; ++j)
      if (himage[j]>0) {
	assert(himage[j] < mbreaks[0]);
	assert(mimage[himage[j]] == j);
      }
  }
  
  // Middle alignments;
  int accepted1 = 0, accepted2 = 0;
  int from1, from2, to1, to2;
  for (i=0; i<breakCnt-1; ++i) {
    assert(hbreaks[i+1] + tuplength - 1 <= hseql && mbreaks[i+1] + tuplength - 1 <= mseql);

    for (j=0; j<tuplength; ++j) {
      himage[hbreaks[i] + j] = mbreaks[i]+j;
      assert(himage[hbreaks[i] + j] <= mseql);

      mimage[mbreaks[i] + j] = hbreaks[i]+j;
    }
    for (j=1; j < hbreaks[i] + tuplength ; ++j) {
      assert(himage[j]);
      if (himage[j]>0) assert(mimage[himage[j]] == j);
    }

    if (hbreaks[i] + tuplength < hbreaks[i+1] && mbreaks[i] + tuplength < mbreaks[i+1]) {
      if (ALIGNVERBOSE) cout << "Aligning " << hbreaks[i] + tuplength << " , " << hbreaks[i+1]-1 
	   << "  ->  "    << mbreaks[i] + tuplength << " , " << mbreaks[i+1]-1;
      
      accepted1 = 0; accepted2 = 0;
      if (extnLength>0 && 
	  hbreaks[i] + tuplength + 2*extnLength < hbreaks[i+1] &&
	  mbreaks[i] + tuplength + 2*extnLength < mbreaks[i+1]) {
	localScore = gappedAlign(hseqint, mseqint, 
				 hbreaks[i] + tuplength, hbreaks[i]+tuplength+extnLength-1,
				 mbreaks[i] + tuplength, mbreaks[i]+tuplength+extnLength-1,
				 match, mismatch, 3*gap, gap, himage + hbreaks[i] + tuplength, mimage + mbreaks[i] + tuplength);
	if (ALIGNVERBOSE) cout << " .." << localScore;
	if (localScore>extnCutoff) accepted1=1;
	localScore = gappedAlign(hseqint, mseqint, 
				 hbreaks[i+1]-extnLength, hbreaks[i+1]-1,
				 mbreaks[i+1]-extnLength, mbreaks[i+1]-1,
				 match, mismatch, 3*gap, gap, himage + hbreaks[i+1]-extnLength, mimage + mbreaks[i+1]-extnLength);
	if (ALIGNVERBOSE) cout << " .." << localScore;
	if (localScore>extnCutoff) accepted2=1;
      }
      if (accepted1) {
	from1 = hbreaks[i]+tuplength+extnLength;
	from2 = mbreaks[i]+tuplength+extnLength;
	if (ALIGNVERBOSE) cout << "\t +";
      }
      else {
	from1 = hbreaks[i] + tuplength;
	from2 = mbreaks[i] + tuplength;
	if (ALIGNVERBOSE) cout << "\t -";
      }
      if (accepted2) {
	to1 = hbreaks[i+1]-extnLength-1;
	to2 = mbreaks[i+1]-extnLength-1;
	if (ALIGNVERBOSE) cout << "\t +" << endl;
      }
      else {
	to1 = hbreaks[i+1]-1;
	to2 = mbreaks[i+1]-1;
	if (ALIGNVERBOSE) cout << "\t -" << endl;
      }
      for (j = hbreaks[i]+tuplength; j < from1; ++j) if (himage[j]>=0) himage[j] += mbreaks[i] + tuplength;
      for (j = mbreaks[i]+tuplength; j < from2; ++j) if (mimage[j]>=0) mimage[j] += hbreaks[i] + tuplength;

      for (j=from1; j <= to1; ++j) himage[j] = -1;
      for (j=from2; j <= to2; ++j) mimage[j] = -1;
      
      for (j = to1+1; j < hbreaks[i+1]; ++j) if (himage[j] >= 0) himage[j] += to2+1;
      for (j = to2+1; j < mbreaks[i+1]; ++j) if (mimage[j] >= 0) mimage[j] += to1+1;

      if (from1<=to1 && from2<=to2) gappedAlign(hseqint, mseqint, from1, to1, from2, to2,
						match, mismatch, 2*gap, gap, himage + from1, mimage + from2);
      for (j = from1; j <= to1; ++j) if (himage[j] >= 0) himage[j] += from2;
      for (j = from2; j <= to2; ++j) if (mimage[j] >= 0) mimage[j] += from1;
      for (j=hbreaks[i]+tuplength; j<hbreaks[i+1]; ++j)
	if (himage[j]>=0) {
	  assert(himage[j] < mbreaks[i+1]);
	  assert(mimage[himage[j]] == j);
	}
    }
  }
  // Last alignment;
  for (j=0; j<tuplength; ++j) {
    himage[hbreaks[i]+j] = mbreaks[i]+j;
    mimage[mbreaks[i]+j] = hbreaks[i]+j;
  }
  for (j=1; j<=hseql; ++j) assert(himage[j]);
  
  if (hbreaks[i] + tuplength <= hseql && mbreaks[i] + tuplength <= mseql) {
    if (ALIGNVERBOSE) cout << "Aligning " << hbreaks[i] + tuplength << " , " << hseql 
			   << "  ->  "    << mbreaks[i] + tuplength << " , " << mseql;
    extended = 0;
    if (extnLength>0 && 
	hbreaks[i] + tuplength + extnLength-1 <= hseql &&
	mbreaks[i] + tuplength + extnLength-1 <= mseql) {
      localScore = gappedAlign(hseqint, mseqint, hbreaks[i]+tuplength, hbreaks[i]+tuplength+extnLength-1,
			       mbreaks[i]+tuplength, mbreaks[i]+tuplength+extnLength-1,
			       match, mismatch, 3*gap, gap, himage+hbreaks[i]+tuplength, mimage+mbreaks[i]+tuplength);
      if (localScore >= extnCutoff) {
	extended = 1;
	for (j = hbreaks[i] + tuplength; j < hbreaks[i] + tuplength + extnLength; ++j) if (himage[j] >= 0) himage[j] += mbreaks[i] + tuplength;
	for (j = mbreaks[i] + tuplength; j < mbreaks[i] + tuplength + extnLength; ++j) if (mimage[j] >= 0) mimage[j] += hbreaks[i] + tuplength;

	if (hbreaks[i]+tuplength+extnLength<= hseql && mbreaks[i]+tuplength+extnLength <= mseql)
	  gappedAlign(hseqint, mseqint, hbreaks[i]+tuplength+extnLength, hseql, mbreaks[i]+tuplength+extnLength, mseql,
		      match, mismatch, 2*gap, gap, himage+hbreaks[i]+tuplength+extnLength, mimage+mbreaks[i]+tuplength+extnLength);
	for (j = hbreaks[i] + tuplength + extnLength; j<=hseql; ++j) if (himage[j] >= 0) himage[j] += mbreaks[i] + tuplength + extnLength;
	for (j = mbreaks[i] + tuplength + extnLength; j<=mseql; ++j) if (mimage[j] >= 0) mimage[j] += hbreaks[i] + tuplength + extnLength;
      }
    }


    if (!extended) {
      for (j=hbreaks[i]+tuplength; j<=hseql; ++j) himage[j] = -1;
      for (j=mbreaks[i]+tuplength; j<=mseql; ++j) mimage[j] = -1;

      gappedAlign(hseqint, mseqint, hbreaks[i]+tuplength, hseql, mbreaks[i]+tuplength, mseql,
		  match, mismatch, 2*gap, gap, himage+ hbreaks[i] + tuplength, mimage + mbreaks[i] + tuplength);
      for (j = hbreaks[i] + tuplength; j <= hseql; ++j) if (himage[j] >= 0) himage[j] += mbreaks[i] + tuplength;
      for (j = mbreaks[i] + tuplength; j <= mseql; ++j) if (mimage[j] >= 0) mimage[j] += hbreaks[i] + tuplength;
    }
  }



  for (i=1; i<=hseql; ++i) {
    assert(himage[i]);
    if (himage[i]>0) assert(mimage[himage[i]]==i);
  }
  
  delete[] hseqint;
  delete[] mseqint;

  return 1;
}


int exactMatchFind(FilterSequence *hseq, FilterSequence *mseq, int tlength,
		   int *htuples, int *hidx, int &hl,
		   int *mtuples, int *midx, int &ml) {

  if (tlength>12) {
    warn("Length too long for  exactMatchFind(..), calling exactLongMatchFind(..)");
    return exactLongMatchFind(hseq, mseq, tlength, htuples, hidx, hl, mtuples, midx, ml);
  }
  int i;
  int hseqlen = hseq->get_length(), mseqlen = mseq->get_length();

  int *htuplesAll = new int[hseqlen+1]; arrayZero(htuplesAll,hseqlen+1);      
  int *mtuplesAll = new int[mseqlen+1]; arrayZero(mtuplesAll,mseqlen+1);
  
  int ttl = power(4,tlength);
  int *tt = new int[ttl]; arrayZero(tt,ttl);
  
  for (i=1; i <= hseqlen; ++i) assert(hseq->get(i) >=0 && hseq->get(i) <= 3);
  for (i=1; i <= mseqlen; ++i) assert(mseq->get(i) >=0 && mseq->get(i) <= 3);

  for (i=1; i <= hseqlen-tlength; ++i) htuplesAll[i] = patt_to_index(*hseq,i,tlength);
  for (i=1; i <= mseqlen-tlength; ++i) mtuplesAll[i] = patt_to_index(*mseq,i,tlength);
  for (i=1; i <= hseqlen-tlength; ++i) tt[htuplesAll[i]] = 1;

  hl = ml = 0;
  
  for (i=1; i <= mseqlen-tlength; ++i)
    if (mtuplesAll[i] >= 0 && mtuplesAll[i] < ttl && tt[mtuplesAll[i]]) { tt[mtuplesAll[i]] = 2; ml++; }
    else mtuplesAll[i] = -1;
  
  for (i=1; i<hseqlen-tlength+1; ++i)
    if (htuplesAll[i] >= 0 && htuplesAll[i] < ttl && tt[htuplesAll[i]] == 2) hl++;
    else htuplesAll[i] = -1;  

  int hptr = 0, mptr = 0;

  for (i=1; i <= hseqlen-tlength; ++i)
    if (htuplesAll[i] >= 0) {
      htuples[hptr] = htuplesAll[i];
      hidx[hptr]    = i;
      hptr++;
    }

  for (i=1; i <= mseqlen-tlength; ++i)
    if (mtuplesAll[i]>=0) {
      mtuples[mptr] = mtuplesAll[i];
      midx[mptr]    = i;
      mptr++;
    }
  assert(hptr == hl && mptr == ml);

  delete[] htuplesAll;
  delete[] mtuplesAll;
  delete[] tt;
  return 1;
}

int exactLongMatchFind(FilterSequence *hseq, FilterSequence *mseq, int tlength,
		       int *htuples, int *hidx, int &hl,
		       int *mtuples, int *midx, int &ml) {

  int i,j;
  int hseqlen = hseq->get_length(), mseqlen = mseq->get_length();


  HugeTupleTable *ltt = new HugeTupleTable();
  int tupIdxcnt = (tlength-11)/12 + ( (tlength-11)%12 ? 1 : 0 );
  int lastIdxl  = (tlength-11)%12 ? (tlength-11)%12 : 12;

  ltt->set_lengths(11, 12, tupIdxcnt);

  int **htuplesAll = new (int*)[tupIdxcnt+1];
  int **mtuplesAll = new (int*)[tupIdxcnt+1];
  
  for (i=0; i<=tupIdxcnt; ++i) {
      htuplesAll[i] = new int[hseqlen+1]; arrayZero(htuplesAll[i], hseqlen+1);
      mtuplesAll[i] = new int[mseqlen+1]; arrayZero(mtuplesAll[i], mseqlen+1);
  }  

  for (i=1; i <= hseqlen; ++i) assert(hseq->get(i) >=0 && hseq->get(i) <= 3);
  for (i=1; i <= mseqlen; ++i) assert(mseq->get(i) >=0 && mseq->get(i) <= 3);

  for (i=1; i <= hseqlen-tlength; ++i) {
    htuplesAll[0][i] = (int)patt_to_index(*hseq, i,    11);
    if (i>136040 && i < 136045) cout << " - to create human tuple " << i << " ... ";
    for (j=0; j<tupIdxcnt-1; ++j)
      htuplesAll[ j+1 ][i] = patt_to_index(*hseq, i+11 + j*12, 12);
    htuplesAll[ j+1 ][i]   = patt_to_index(*hseq, i+11 + j*12, lastIdxl);
  }
  for (i=1; i <= mseqlen-tlength; ++i) {
    mtuplesAll[0][i] = patt_to_index(*mseq, i, 11);
    for (j=0; j<tupIdxcnt-1; ++j)
      mtuplesAll[ j+1 ][i] = patt_to_index(*mseq, i+11 + j*12, 12);
    mtuplesAll[ j+1 ][i]   = patt_to_index(*mseq, i+11 + j*12, lastIdxl);
  }
  for (i=1; i <= hseqlen-tlength; ++i) {
    int indices[100]; for (j=0; j<tupIdxcnt; ++j) indices[j] = htuplesAll[j+1][i];
    if (!ltt->isHit(htuplesAll[0][i], indices)) {
      ltt->add_entry(htuplesAll[0][i], indices, i);
      int datum;
      assert(ltt->getDatum(htuplesAll[0][i], indices, datum));
      assert(datum == i);
    }
  }
  hl = ml = 0;
  
  for (i=1; i <= mseqlen-tlength; ++i)
    if (mtuplesAll[0][i] >= 0) {
      int indices[100]; for (j=0; j<tupIdxcnt; ++j) indices[j] = mtuplesAll[j+1][i];
      if (ltt->isHit(mtuplesAll[0][i], indices)) {
	int datum;
	assert(ltt->getDatum(mtuplesAll[0][i], indices, datum));
	assert(datum > 0);
	ltt->add_entry(mtuplesAll[0][i], indices);
	int datum2;
	assert(ltt->isHit(mtuplesAll[0][i], indices, 2));
	assert(ltt->getDatum(mtuplesAll[0][i], indices, datum2));
	assert(datum==datum2);
	ml++;
      }
      else mtuplesAll[0][i] = -1;
    }
    else mtuplesAll[0][i] = -1;
  
  for (i=1; i<hseqlen-tlength+1; ++i)
    if (htuplesAll[0][i] >= 0) {
      int indices[100]; for (j=0; j<tupIdxcnt; ++j) indices[j] = htuplesAll[j+1][i];
      if (ltt->isHit(htuplesAll[0][i], indices, 2)) hl++;
      else htuplesAll[0][i] = -1;
    }
    else htuplesAll[0][i] = -1;

  int hptr = 0, mptr = 0;
  for (i=1; i <= hseqlen-tlength; ++i)
    if (htuplesAll[0][i] >= 0) {
      int indices[100]; for (j=0; j<tupIdxcnt; ++j) indices[j] = htuplesAll[j+1][i];
      assert( ltt->getDatum( htuplesAll[0][i], indices,  htuples[hptr] ) );
      assert(htuples[hptr]>0);
      
      hidx[hptr]    = i;
      hptr++;
    }
  
  for (i=1; i <= mseqlen-tlength; ++i)
    if (mtuplesAll[0][i]>=0) {
      int indices[100]; for (j=0; j<tupIdxcnt; ++j) indices[j] = mtuplesAll[j+1][i];
      assert( ltt->getDatum( mtuplesAll[0][i], indices,  mtuples[mptr] ) );      
      assert(mtuples[mptr]>0);
      
      midx[mptr]    = i;
      mptr++;
    }
  assert(hptr == hl && mptr == ml);

  for (j=0; j<=tupIdxcnt; ++j) {
    delete[] htuplesAll[j];
    delete[] mtuplesAll[j];
  }
  delete[] htuplesAll;
  delete[] mtuplesAll;
  delete ltt;
  return 1;
}

double findmap(int *seq1, int *seq2, int from1, int to1, int from2, int to2,  
	     double match, double mismatch, double gap, int *imagefirst, 
	     int *imagesecond, double *col1, double *col2, double *prebest, 
	     double *suffbest){
  
  int i,j, k;
  int lengthfirst = to1 - from1 + 1;
  int lengthsecond = to2 - from2 + 1;
  if(lengthfirst==0 || lengthsecond == 0)
    return 0;
  int m=(int)ceil(((double)lengthfirst)/2)-1+from1;
    
  int mu=m-from1+1;
  //  cout << "m: " << m << endl;
  //cout << "mu: " << mu << endl;
  //cout << "from1: " << from1 << endl;
  //cout << "to1: " << to1 << endl;
  //cout << "from2: " << from2 << endl;
  //cout << "to2: " << to2 << endl;
  
  double scorek[2]; int maxindex=0; int gapflag=0; int tempgapflag=0; double maxscore=-Infinity;
  double up, left, upleft;
  
  k=lengthsecond;
  // Prefix Calculation
  // Boundary Conditions:
  prebest[0]=gap*(m-from1);
  
  for (i=0; i<=mu-1; ++i)
    col1[i]=gap*i;
   for (j=1; j<=k; ++j){
    col2[0]=gap*j;
    for (i=1; i<=mu-1; ++i){
      up=col2[i-1]+gap;
      left=col1[i]+gap;
      upleft=col1[i-1]+( (seq1[from1+i-1]==seq2[from2+j-1]) ? match : mismatch);
      col2[i]=MAX(up,left,upleft);
      col1[i-1]=col2[i-1];
      if (i==mu-1 && j!=k)
	col1[mu-1]=col2[mu-1];
    }
    prebest[j]=col2[mu-1]; 
  }
  if (m==from1)
    for (j=1; j<=k; ++j)
      prebest[j]=gap*j;
  
  
  k=0;
  // Suffix Calculation
  // Boundary Conditions:
  suffbest[0]=gap*(to1-m);
  for (i=0; i<=to1-m; ++i)
    col1[i]=gap*i;
  for (j=1; j<=lengthsecond-k; ++j){
    col2[0]=gap*j;
    for (i=1; i<=to1-m; ++i){
      up=col2[i-1]+gap;
      left=col1[i]+gap;
      upleft=col1[i-1]+( (seq1[to1-i+1]==seq2[to2-j+1]) ? match : mismatch);
      col2[i]=MAX(up,left,upleft);
      col1[i-1]=col2[i-1];
      if (i==to1-m && j!=lengthsecond-k)
	col1[to1-m]=col2[to1-m];
    }
    suffbest[j]=col2[to1-m];
  }
  if (m==to1)
    for (j=1; j<=lengthsecond; ++j)
      suffbest[j]=gap*j;
  
  for (k=0; k<=lengthsecond; ++k){
    if (k>0){
      scorek[0]=prebest[k-1]+suffbest[lengthsecond-k]+((seq1[m]==seq2[k+from2-1]) ? match : mismatch);
      scorek[1]=prebest[k]+suffbest[lengthsecond-k]+gap;
    }
    else if (k==0){
      scorek[0]=-Infinity;
      scorek[1]=prebest[0]+suffbest[lengthsecond]+gap;
    }
    
    if (scorek[0]>=scorek[1])
      tempgapflag=0;
    else
      tempgapflag=1;
    if (scorek[tempgapflag] > maxscore){
      maxscore=scorek[tempgapflag];
      maxindex=k+from2-1;
      gapflag=tempgapflag;
    }
  }
  if (gapflag==0) {
    assert(maxindex>=0);
    assert(imagefirst[m]==-1);
    //      cout << " ### " << m-1 << " -->> " << maxindex-1 << endl;
    imagefirst[m]=maxindex;
    assert(imagesecond[imagefirst[m]]==-1);
    imagesecond[imagefirst[m]]=m;
  }
   else imagefirst[m]=-1;
  
  if (m-1>=from1 && from2 <=maxindex-1+gapflag)
    findmap(seq1, seq2, from1, m-1, from2, maxindex-1+gapflag, match, mismatch, gap, imagefirst, imagesecond, col1, col2, prebest, suffbest);
  if (m+1 <= to1 && maxindex+1 <=to2)
    findmap(seq1, seq2, m+1, to1, maxindex+1, to2,  match, mismatch, gap, imagefirst, imagesecond, col1, col2, prebest, suffbest);
  return maxscore;
  
}

double findmapUseExtensions(int *seq1, int *seq2, int from1, int to1, int from2, int to2,  
			    double match, double mismatch, double gap, int *imagefirst, 
			    int *imagesecond, double *col1, double *col2, double *prebest, double *suffbest,
			    int *hseqInt, int *mseqInt, int *hptrs, int *mptrs, int hseql, int mseql, int extnLength, int tupLength) {
  
  int i,j, k;
  int lengthfirst = to1 - from1 + 1;
  int lengthsecond = to2 - from2 + 1;
  if(lengthfirst==0 || lengthsecond == 0)
    return 0;
  int m=(int)ceil(((double)lengthfirst)/2)-1+from1;
    
  int mu=m-from1+1;
  //  cout << "m: " << m << endl;
  //cout << "mu: " << mu << endl;
  //cout << "from1: " << from1 << endl;
  //cout << "to1: " << to1 << endl;
  //cout << "from2: " << from2 << endl;
  //cout << "to2: " << to2 << endl;
  
  double scorek[2]; int maxindex=0; int gapflag=0; int tempgapflag=0; double maxscore=-Infinity;
  double up, left, upleft;
  
  k=lengthsecond;
  // Prefix Calculation
  // Boundary Conditions:
  prebest[0]=gap*(m-from1);
  
  for (i=0; i<=mu-1; ++i)
    col1[i]=gap*i;
   for (j=1; j<=k; ++j){
    col2[0]=gap*j;
    for (i=1; i<=mu-1; ++i){
      up=col2[i-1]+gap;
      left=col1[i]+gap;
      if (seq1[from1+i-1]==seq2[from2+j-1]) {
	double alignScoreLeft, alignScoreRight;
	int    *himageLeft = new int[extnLength], *himageRight = new int[extnLength];
	int    *mimageLeft = new int[extnLength], *mimageRight = new int[extnLength];
	arrayZero(himageLeft,  extnLength); arrayZero(himageRight, extnLength);
	arrayZero(mimageLeft,  extnLength); arrayZero(mimageRight, extnLength);

	if (hptrs[from1+i-1] > extnLength && mptrs[from2+j-1] > extnLength)
	  alignScoreLeft = gappedAlign(hseqInt, mseqInt,
				       hptrs[from1+i-1] - extnLength, hptrs[from1+i-1] - 1,
				       mptrs[from2+j-1] - extnLength, mptrs[from2+j-1] - 1,
				       1, -1, -2, -1,
				       himageLeft, mimageLeft);
	else alignScoreLeft = 0;

	if ( (hptrs[from1+i-1] + tupLength + extnLength < hseql) && (mptrs[from2+j-1] + tupLength + extnLength < mseql) )
	  alignScoreRight = gappedAlign(hseqInt, mseqInt,
					hptrs[from1+i-1] + tupLength, hptrs[from1+i-1] + tupLength + extnLength - 1,
					mptrs[from2+j-1] + tupLength, mptrs[from2+j-1] + tupLength + extnLength - 1,
					1, -1, -2, -1,
					himageRight, mimageRight);
	else alignScoreRight = 0;
	
	delete[] himageLeft; delete[] himageRight;
	delete[] mimageLeft; delete[] mimageRight;
	upleft=col1[i-1] + alignScoreLeft + alignScoreRight;
      }
      else upleft = col1[i-1] + mismatch;

      col2[i]=MAX(up,left,upleft);
      col1[i-1]=col2[i-1];
      if (i==mu-1 && j!=k)
	col1[mu-1]=col2[mu-1];
    }
    prebest[j]=col2[mu-1]; 
  }
  if (m==from1)
    for (j=1; j<=k; ++j)
      prebest[j]=gap*j;
  
  
  k=0;
  // Suffix Calculation
  // Boundary Conditions:
  suffbest[0]=gap*(to1-m);
  for (i=0; i<=to1-m; ++i)
    col1[i]=gap*i;
  for (j=1; j<=lengthsecond-k; ++j){
    col2[0]=gap*j;
    for (i=1; i<=to1-m; ++i){
      up=col2[i-1]+gap;
      left=col1[i]+gap;


      if (seq1[to1-i+1]==seq2[to2-j+1]) {
	double alignScoreLeft, alignScoreRight;
	int    *himageLeft = new int[extnLength], *himageRight = new int[extnLength];
	int    *mimageLeft = new int[extnLength], *mimageRight = new int[extnLength];
	arrayZero(himageLeft,  extnLength); arrayZero(himageRight, extnLength);
	arrayZero(mimageLeft,  extnLength); arrayZero(mimageRight, extnLength);

	if (hptrs[to1-i+1] > extnLength && mptrs[to2-j+1] > extnLength)
	  alignScoreLeft = gappedAlign(hseqInt, mseqInt,
				       hptrs[to1-i+1] - extnLength, hptrs[to1-i+1] - 1,
				       mptrs[to2-j+1] - extnLength, mptrs[to2-j+1] - 1,
				       1, -1, -2, -1,
				       himageLeft, mimageLeft);
	else alignScoreLeft = 0;

	if (hptrs[to1+i-1] + tupLength + extnLength < hseql && mptrs[to2+j-1] + tupLength + extnLength < mseql)
	  alignScoreRight = gappedAlign(hseqInt, mseqInt,
					hptrs[to1+i-1] + tupLength, hptrs[to1+i-1] + tupLength + extnLength - 1,
					mptrs[to2+j-1] + tupLength, mptrs[to2+j-1] + tupLength + extnLength - 1,
					1, -1, -2, -1,
					himageRight, mimageRight);
	else alignScoreRight = 0;
	delete[] himageLeft; delete[] himageRight;
	delete[] mimageLeft; delete[] mimageRight;
	upleft=col1[i-1] + alignScoreLeft + alignScoreRight;
      }
      else upleft = col1[i-1] + mismatch;

      col2[i]=MAX(up,left,upleft);
      col1[i-1]=col2[i-1];
      if (i==to1-m && j!=lengthsecond-k)
	col1[to1-m]=col2[to1-m];
    }
    suffbest[j]=col2[to1-m];
  }
  if (m==to1)
    for (j=1; j<=lengthsecond; ++j)
      suffbest[j]=gap*j;
  
  for (k=0; k<=lengthsecond; ++k){
    if (k>0){
      scorek[0]=prebest[k-1]+suffbest[lengthsecond-k]+((seq1[m]==seq2[k+from2-1]) ? match : mismatch);
      scorek[1]=prebest[k]+suffbest[lengthsecond-k]+gap;
    }
    else if (k==0){
      scorek[0]=-Infinity;
      scorek[1]=prebest[0]+suffbest[lengthsecond]+gap;
    }
    
    if (scorek[0]>=scorek[1])
      tempgapflag=0;
    else
      tempgapflag=1;
    if (scorek[tempgapflag] > maxscore){
      maxscore=scorek[tempgapflag];
      maxindex=k+from2-1;
      gapflag=tempgapflag;
    }
  }
  if (gapflag==0) {
    assert(maxindex>=0);
    assert(imagefirst[m]==-1);
    //      cout << " ### " << m-1 << " -->> " << maxindex-1 << endl;
    imagefirst[m]=maxindex;
    assert(imagesecond[imagefirst[m]]==-1);
    imagesecond[imagefirst[m]]=m;
  }
   else imagefirst[m]=-1;
  
  if (m-1>=from1 && from2 <=maxindex-1+gapflag)
    findmapUseExtensions(seq1, seq2, from1, m-1, from2, maxindex-1+gapflag, match, mismatch, gap,
			 imagefirst, imagesecond, col1, col2, prebest, suffbest,
			 hseqInt, mseqInt, hptrs, mptrs, hseql, mseql, extnLength, tupLength);
  if (m+1 <= to1 && maxindex+1 <=to2)
    findmapUseExtensions(seq1, seq2, m+1, to1, maxindex+1, to2,  match, mismatch, gap,
			 imagefirst, imagesecond, col1, col2, prebest, suffbest,
			 hseqInt, mseqInt, hptrs, mptrs, hseql, mseql, extnLength, tupLength);
  return maxscore;
  
}

  
double memalign(int *first, int *second, int fromfirst, int tofirst, int fromsecond, 
	      int tosecond, double match, double mismatch, double gap, 
	      int *imagefirst, int *imagesecond){
  assert(fromfirst<=tofirst);
  assert(fromsecond<=tosecond);
  // Allocate Memory
  int i;
  double score;
  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;
  double *col1 = new double[lengthfirst+1]; arrayZero(col1, lengthfirst+1);
  double *col2 = new double[lengthfirst+1]; arrayZero(col2, lengthfirst+1);
  double *prebest = new double[lengthsecond+1]; arrayZero(prebest, lengthsecond+1);
  double *suffbest = new double[lengthsecond+1]; arrayZero(suffbest, lengthsecond+1);
  
  for (i=0; i<lengthfirst; ++i)
    imagefirst[i]=-1;
  for (i=0; i<lengthsecond; ++i)
    imagesecond[i]=-1;

  score=findmap(first+fromfirst, second+fromsecond, 0, tofirst-fromfirst, 0, tosecond-fromsecond,
	  match, mismatch, gap, imagefirst, imagesecond, col1, col2, prebest, suffbest);
  
  // Delete memory
  delete[] col1;
  delete[] col2;
  delete[] prebest;
  delete[] suffbest;
  
  return score;
}


double memalignUseExtensions(int *first, int *second, int fromfirst, int tofirst, int fromsecond, 
			     int tosecond, double match, double mismatch, double gap, 
			     int *imagefirst, int *imagesecond, int *hseqInt, int *mseqInt,
			     int *hptrs, int *mptrs, int hseql, int mseql, int extnLength, int tupLength){
  assert(fromfirst<=tofirst);
  assert(fromsecond<=tosecond);
  // Allocate Memory
  int i;
  double score;
  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;
  double *col1 = new double[lengthfirst+1]; arrayZero(col1, lengthfirst+1);
  double *col2 = new double[lengthfirst+1]; arrayZero(col2, lengthfirst+1);
  double *prebest = new double[lengthsecond+1]; arrayZero(prebest, lengthsecond+1);
  double *suffbest = new double[lengthsecond+1]; arrayZero(suffbest, lengthsecond+1);
  
  for (i=0; i<lengthfirst; ++i)
    imagefirst[i]=-1;
  for (i=0; i<lengthsecond; ++i)
    imagesecond[i]=-1;
  
  score = findmapUseExtensions(first, second, fromfirst, tofirst, fromsecond, tosecond,
			       match, mismatch, gap, imagefirst, imagesecond, 
			       col1, col2, prebest, suffbest,
			       hseqInt, mseqInt, hptrs, mptrs,
			       hseql, mseql, extnLength, tupLength);  

  // Delete memory
  delete[] col1;
  delete[] col2;
  delete[] prebest;
  delete[] suffbest;
  
  return score;
}

double align(int *first, int *second, int fromfirst, int tofirst, int fromsecond, 
	     int tosecond, double match, double mismatch, double gap, 
	     int *imagefirst, int *imagesecond){
  
  // Performs a global alignment on two sequences. M contains the scores, P contains
  // the pointers. The encoding is 0=up, 1=left, 2=upleft.
  // The input is two sequences, and positions within them to align. The answer is returned
  // in imagefirst and imagesecond (zero based).
  
  assert(fromfirst<=tofirst);
  assert(fromsecond<=tosecond);
  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;
  int i=0,j=0;
  double up=0, left=0, upleft=0;
  double score=0;
  double **MatrixM=new (double*)[lengthfirst+2];
  for (i=0; i<lengthfirst+2; ++i) MatrixM[i]=new double[lengthsecond+2];
  double **MatrixP=new (double*)[lengthfirst+2];
  for (i=0; i<lengthfirst+2; ++i) MatrixP[i]=new double[lengthsecond+2];

  // Boundary Conditions:
  for (i=0; i<lengthfirst+1; ++i){
    MatrixM[i][0]=gap*i;
    MatrixP[i][0]=0;
  }
  for (j=0; j<lengthsecond+1; ++j){
    MatrixM[0][j]=gap*j;
    MatrixP[0][j]=1;
  }
  for (i=1; i<lengthfirst+1; ++i){
    for (j=1; j<lengthsecond+1; ++j){
      up=MatrixM[i-1][j]+gap;
      left=MatrixM[i][j-1]+gap;
      upleft=MatrixM[i-1][j-1]+( (first[fromfirst+i-1]==second[fromsecond+j-1]) ? match : mismatch);
      MatrixM[i][j]=MAX(up,left,upleft);
      if (upleft>=up && upleft>=left) MatrixP[i][j] = 2;
      else if (up>=left)              MatrixP[i][j] = 0;
      else                            MatrixP[i][j] = 1;
    }
  }
   // find optimal alignment and fill in the arrays imagefirst and imagesecond
  i=lengthfirst;
  j=lengthsecond;
  score=MatrixM[i][j];

  while (i>0 || j>0) {
    if (MatrixP[i][j]==0){
      imagefirst[i-1]=-1;
      i--;
     continue;
    }
    if (MatrixP[i][j]==1){
      imagesecond[j-1]=-1;
      j--;
      continue;
    }
    if (MatrixP[i][j]==2){
      imagefirst[i-1]=j-1;
      imagesecond[j-1]=i-1;
      i--;
      j--;
      continue;
    }
  }
  int lastImageFirst = -1;
  for (i=0; i<lengthfirst; ++i) {
    if (imagefirst[i] >=0) {
      lastImageFirst = imagefirst[i];
      assert(imagesecond[imagefirst[i]] == i);
    }
  }
   // cleanup
  for (i=lengthfirst+1; i>=0; --i) {
    delete[] MatrixM[i]; MatrixM[i] = 0;
  } delete[] MatrixM; MatrixM = 0;
  
  for (i=lengthfirst+1; i>=0; --i) {
    delete[] MatrixP[i]; MatrixP[i] = 0;
  } delete[] MatrixP; MatrixP = 0;

  return score;
}

enum alignPtrs { toD0 = 1, toD1 = 2, toU0 = 3, toU1 = 4, toL0 = 5, toL1 = 6 };


double gappedAlign(int *first, int *second, int fromfirst, int tofirst, int fromsecond, int tosecond,
		   double match, double mismatch, double gapOpen, double gap, int *imagefirst, int *imagesecond) {
  
  // Performs a global alignment on two sequences. M contains the scores, P contains
  // the pointers. The encoding is 0=up, 1=left, 2=upleft.
  // The input is two sequences, and positions within them to align. The answer is returned
  // in imagefirst and imagesecond (zero based).
 
  if ( (tofirst-fromfirst) * (tosecond-fromsecond) > 1024 * 2048 ) {
    warn("alignment too big for gappedAlign. Using memalign instead !");
    return memalign(first, second, fromfirst, tofirst, fromsecond, tosecond, match, mismatch, gap, imagefirst, imagesecond);
  }
 
  assert(fromfirst<=tofirst);
  assert(fromsecond<=tosecond);

  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;
  int i=0,j=0;

  double score=0;
  double ***MatrixM=new (double**)[lengthfirst+2];

  for (i=0; i<lengthfirst+2; ++i) {
    MatrixM[i]=new (double*)[lengthsecond+2];
    for (j=0; j<lengthsecond+2; ++j) {
      MatrixM[i][j] = new double[2];
      MatrixM[i][j][0] = MatrixM[i][j][1] = 0;
    }
  }
  int ***MatrixP=new (int**)[lengthfirst+2];
  for (i=0; i<lengthfirst+2; ++i) {
    MatrixP[i]=new (int*)[lengthsecond+2];
    for (j=0; j<lengthsecond+2; ++j) {
      MatrixP[i][j] = new int[2];
      MatrixP[i][j][0] = MatrixP[i][j][1] = 0;
    }
  }

  // Boundary Conditions:
  MatrixM[0][0][0] = 0;
  MatrixM[0][0][1] = -Infinity;
  MatrixP[0][0][0] = MatrixP[0][0][1] = 0;

  for (i=1; i<lengthfirst+1; ++i){
    MatrixM[i][0][0] = -Infinity; MatrixM[i][0][1] = gap*i + gapOpen-gap;
    MatrixP[i][0][0] = toU1;      MatrixP[i][0][1] = toU1;
  }
  MatrixP[1][0][0] = toU0;

  for (j=1; j<lengthsecond+1; ++j){
    MatrixM[0][j][0] = -Infinity; MatrixM[0][j][1] = gap*j + gapOpen-gap;
    MatrixP[0][j][0] = toL1;      MatrixP[0][j][1] = toL1;
  }
  MatrixP[0][1][0] = toL0;

  for (i=1; i<lengthfirst+1; ++i){
    for (j=1; j<lengthsecond+1; ++j){

      double U0 = MatrixM[i-1][ j ][0],   U1 = MatrixM[i-1][ j ][1];
      double L0 = MatrixM[ i ][j-1][0],   L1 = MatrixM[ i ][j-1][1];
      double D0 = MatrixM[i-1][j-1][0],   D1 = MatrixM[i-1][j-1][1];
      
      double d = (first[fromfirst+i-1]==second[fromsecond+j-1]) ? match : mismatch;
      double g = gapOpen;
      double e = gap;
      
      MatrixM[i][j][0] = MAX( D0 , D1 ) + d;
      
      if (D0>=D1) MatrixP[i][j][0] = toD0;
      else        MatrixP[i][j][0] = toD1;

      MatrixM[i][j][1] = MAX( MAX( U0,L0 ) + g   ,   MAX( U1,L1 ) + e );

      if      (U0 >= L0 && U0+g >= MAX(U1,L1)+e)   MatrixP[i][j][1] = toU0;
      else if (U1 >= L1 && U1+e >=     L0    +g)   MatrixP[i][j][1] = toU1;
      else if (            L0+g >=     L1    +e)   MatrixP[i][j][1] = toL0;
      else                                         MatrixP[i][j][1] = toL1;
    }
  }
   // find optimal alignment and fill in the arrays imagefirst and imagesecond
  i=lengthfirst;
  j=lengthsecond;
  int l = (MatrixM[i][j][0] >= MatrixM[i][j][1]) ? 0 : 1;
  
  score=MatrixM[i][j][l];

  while (i>0 || j>0) {

    switch( MatrixP[i][j][l] ) {

    case toU0:
      imagefirst[i-1] = -1;
      l = 0;
      i--;
      break;

    case toU1:
      imagefirst[i-1] = -1;
      l = 1;
      i--;
      break;
    
    case toL0:
      imagesecond[j-1] = -1;
      l = 0;
      j--;
      break;

    case toL1:
      imagesecond[j-1] = -1;
      l = 1;
      j--;
      break;

    case toD0:
      imagefirst[i-1]  = j-1;
      imagesecond[j-1] = i-1;
      l = 0;
      i--;
      j--;
      break;
    
    case toD1:
      imagefirst[i-1]  = j-1;
      imagesecond[j-1] = i-1;
      l = 1;
      i--;
      j--;
      break;
      
    default: assert(0);
    }
  }
  int lastImageFirst = -1;
  for (i=0; i<lengthfirst; ++i) {
    if (imagefirst[i] >=0) {
      lastImageFirst = imagefirst[i];
      assert(imagesecond[imagefirst[i]] == i);
    }
  }
   // cleanup
  for (i=lengthfirst+1; i>=0; --i) {
    for (j=lengthsecond+1; j>=0; --j) delete[] MatrixM[i][j];
    delete[] MatrixM[i]; MatrixM[i] = 0;
  } delete[] MatrixM; MatrixM = 0;
  
  for (i=lengthfirst+1; i>=0; --i) {
    for (j=lengthsecond+1; j>=0; --j) delete[] MatrixP[i][j];
    delete[] MatrixP[i]; MatrixP[i] = 0;
  } delete[] MatrixP; MatrixP = 0;
  
  return score;
}


void constraintsalign(int *first, int *second, int length1, int length2, 
		      double match, double mismatch, double gap, int *imagefirst, 
		      int *imagesecond, int *constraints1, int *constraints2, 
		      int constraintslength, int acceptNegativeAlignments){
  int i, tempi;
  for (i=0; i<length1; ++i)
    imagefirst[i]=-1;
  for (i=0; i<length2; ++i)
    imagesecond[i]=-1;
  
  int from1=0, from2=0, to1=0, to2=0;
  double piecescore=-1; int difflength;
  double *col1 = new double[length1+1]; arrayZero(col1, length1+1);
  double *col2 = new double[length1+1]; arrayZero(col2, length1+1);
  double *prebest = new double[length2+1]; arrayZero(prebest, length2+1);
  double *suffbest = new double[length2+1]; arrayZero(suffbest, length2+1);
  
  // begin alignment
  for (i=0; i<constraintslength; ++i){
    to1=constraints1[i]-1;
    to2=constraints2[i]-1;
    imagefirst[constraints1[i]]=constraints2[i];
    imagesecond[constraints2[i]]=constraints1[i];
    if (!((from1==constraints1[i]) || (from2==constraints2[i]))){
      if (i==0 && (from1 > to1 || from2 > to2)) continue;
      assert(from1<=to1 && from2 <= to2);
      
      piecescore=findmap(first, second, from1, to1, from2, to2, match, mismatch, gap, imagefirst, imagesecond, col1, col2, prebest, suffbest);
      
      //	cout << "psc1: " << piecescore << " ";
      if (!acceptNegativeAlignments) {
	piecescore = tupleAlignmentScore(first, second, from1, to1, from2, to2, match, mismatch, gap, 0.8, 1.5, imagefirst, imagesecond);
	//	cout << "psc2: " << piecescore << "       ";
	difflength=abs((to1-from1) - (to2-from2));
        if (ALIGNVERBOSE)	cout << from1 << "/" << to1 << " - " << from2 << "/" << to2 << "  piecescore: " << piecescore << "; difflength: " << difflength << endl;
	if (piecescore<0){
	  for (tempi=from1; tempi<=to1; ++tempi)
	    imagefirst[tempi]=-1;
	  for (tempi=from2; tempi<=to2; ++tempi)
	    imagesecond[tempi]=-1;
	}
      }
    }
    from1=constraints1[i]+1;
    from2=constraints2[i]+1;
  }
  if (from1 <= length1-1 && from2 <= length2-1) {
    to1=length1-1;
    to2=length2-1;
    findmap(first, second, from1, to1, from2, to2, match, mismatch, gap, imagefirst, imagesecond, col1, col2, prebest, suffbest);
  }
  delete[] col1;
  delete[] col2;
  delete[] prebest;
  delete[] suffbest;
}
void constraintsalignUseExtensions(int *first, int *second, int length1, int length2, 
				   double match, double mismatch, double gap, int *imagefirst, 
				   int *imagesecond, int *constraints1, int *constraints2, 
				   int constraintslength, int acceptNegativeAlignments,
				   int *hseqInt, int *mseqInt, int *hptrs, int *mptrs,
				   int hseql, int mseql, int extnLength, int tupLength) {
  int i;
  for (i=0; i<length1; ++i)
    imagefirst[i]=-1;
  for (i=0; i<length2; ++i)
    imagesecond[i]=-1;
  
  int from1=0, from2=0, to1=0, to2=0;
  double piecescore=-1; 
  double *col1     = new double[length1+1]; arrayZero(col1, length1+1);
  double *col2     = new double[length1+1]; arrayZero(col2, length1+1);
  double *prebest  = new double[length2+1]; arrayZero(prebest, length2+1);
  double *suffbest = new double[length2+1]; arrayZero(suffbest, length2+1);
  
  // begin alignment
  for (i=0; i<constraintslength; ++i){
    to1=constraints1[i]-1;
    to2=constraints2[i]-1;
    imagefirst[constraints1[i]]=constraints2[i];
    imagesecond[constraints2[i]]=constraints1[i];
    if (!((from1==constraints1[i]) || (from2==constraints2[i]))){
      if (i==0 && (from1 > to1 || from2 > to2)) continue;
      assert(from1<=to1 && from2 <= to2);
      
      piecescore = findmapUseExtensions(first, second, from1, to1, from2, to2,
					match, mismatch, gap, imagefirst, imagesecond, 
					col1, col2, prebest, suffbest,
					hseqInt, mseqInt, hptrs, mptrs,
					hseql, mseql, extnLength, tupLength);
      if (ALIGNVERBOSE) cout << "Constraints Aligning:\t" << from1 << "," << to1 << "  (" << hptrs[from1] << "," << hptrs[to1]
			     << ")  ---> "                << from2 << "," << to2 << "  (" << mptrs[from2] << "," << mptrs[to2] << ")"
			     << "\t" << piecescore << endl;
    }
    from1=constraints1[i]+1;
    from2=constraints2[i]+1;
  }
  if (from1 <= length1-1 && from2 <= length2-1) {
    to1=length1-1;
    to2=length2-1;
    findmapUseExtensions(first, second, from1, to1, from2, to2, match, mismatch, gap,
			 imagefirst, imagesecond, col1, col2, prebest, suffbest,
			 hseqInt, mseqInt, hptrs, mptrs, hseql, mseql, extnLength, tupLength);
  }
  delete[] col1;
  delete[] col2;
  delete[] prebest;
  delete[] suffbest;
}

int phaseAlign(FilterSequence *hseq, FilterSequence *mseq, int *hseqInt, int *mseqInt,
	       int phaseCount, int *tupLengths, int *extnLengths, int *extnCutoffs,
	       int *humImg, int *mouseImg) {
  assert(phaseCount>0);
  int i=0,j=0;
  for (i=0; i<phaseCount; ++i) {
    assert(tupLengths[i]>0); assert(extnLengths[i]>0);
    if (i) assert(tupLengths[i] < tupLengths[i-1]);
  }

  int ph = 0, hseql = hseq->get_length(), mseql = mseq->get_length(), currLength, lastConstrPos;
  double alignScoreLeft, alignScoreRight;
 
  for (i=0; i<phaseCount; ++i) {
    assert(tupLengths[i]>0);
    if (i) assert(tupLengths[i] < tupLengths[i-1]);
  }

  int **htuples = new (int*)[phaseCount], **hidx = new (int*)[phaseCount];
  int **mtuples = new (int*)[phaseCount], **midx = new (int*)[phaseCount];
  for (i=0; i<phaseCount; ++i) {
    htuples[i] = new int[hseql+1]; arrayZero(htuples[i], hseql+1);
    mtuples[i] = new int[mseql+1]; arrayZero(mtuples[i], mseql+1);
    hidx[i]    = new int[hseql+1]; arrayZero(hidx[i],    hseql+1);
    midx[i]    = new int[mseql+1]; arrayZero(midx[i],    mseql+1);
  }
  int *hl = new int[phaseCount]; arrayZero(hl, phaseCount);
  int *ml = new int[phaseCount]; arrayZero(ml, phaseCount);
  
  for (i=0; i<phaseCount; ++i) {
    exactMatchFind(hseq, mseq, tupLengths[i], htuples[i], hidx[i], hl[i], mtuples[i], midx[i], ml[i]);
    if (ALIGNVERBOSE) cout << "Found " << hl[i]  << " , " << ml[i]  << " " << tupLengths[i]  << "-tuple matches " << endl;
    if (tupLengths[i] >= 23) {
      cout << endl << endl;
      for (j=0; j<hl[i]; ++j) {
	cout << hidx[i][j] << ": " << htuples[i][j] << "    ";
	if ( !(j % 10) ) cout << endl;
      }
      cout << endl << endl;
      for (j=0; j<ml[i]; ++j) {
	cout << midx[i][j] << ": " << mtuples[i][j] << "    ";
	if ( !(j % 8) ) cout << endl;
      }
      cout << endl << endl;
    }
  }
  int **htupImg = new (int*)[phaseCount], **hbreaks = new (int*)[phaseCount];
  int **mtupImg = new (int*)[phaseCount], **mbreaks = new (int*)[phaseCount];
  for (i=0; i<phaseCount; ++i) {
    htupImg[i] = new int[ hl[i] ]; arrayInit(-1, htupImg[i], hl[i]);
    hbreaks[i] = new int[ hl[i] ]; arrayInit(-1, hbreaks[i], hl[i]);
    mtupImg[i] = new int[ ml[i] ]; arrayZero(mtupImg[i], ml[i]);
    mbreaks[i] = new int[ ml[i] ]; arrayZero(mbreaks[i], ml[i]);
  }
  int *breakCnt        = new int[phaseCount]; arrayZero(breakCnt, phaseCount);
  int *hconstrExplicit = new int[hseql+1];
  int *hinverseIdx     = new int[hseql+1];
  int *minverseIdx     = new int[mseql+1];
  int *hconstraints    = new int[hseql+1];
  int *mconstraints    = new int[mseql+1];
  
  int *himgFixed       = new int[hseql+1]; arrayInit(-2, himgFixed, hseql+1);
  int *mimgFixed       = new int[mseql+1]; arrayInit(-2, mimgFixed, mseql+1);

  int firstPass = 0;
  double firstPassScore = 0;
        
  for (ph = 0; ph<phaseCount; ++ph) {
    assert(extnLengths[ph] > 0);
    if (!firstPass) {
      if (ALIGNVERBOSE) cout << " trying " << tupLengths[0] << "-tuples " << endl;

      /*
      firstPassScore = memalignUseExtensions(htuples[0], mtuples[0], 0, hl[0]-1, 0, ml[0]-1, 1, -1, -1, htupImg[0], mtupImg[0],
					     hseqInt, mseqInt, hidx[0], midx[0], hseql, mseql, extnLengths[0], tupLengths[0]);
      */
      firstPassScore = memalign(htuples[0], mtuples[0], 0, hl[0]-1, 0, ml[0]-1, 1, -1, -1, htupImg[0], mtupImg[0]);
      
      firstPassScore = tupleAlignmentScore(htuples[0], mtuples[0], 0, hl[0]-1, 0, ml[0]-1, 1, -1, -1, 0.8, 1.5, htupImg[0], mtupImg[0]);
      firstPass = 1;
      if (ALIGNVERBOSE) cout << "first pass score: " << firstPassScore << "     tuple size: " << tupLengths[0] << endl;
      continue;
    }
    assert(hl[ph] >= hl[ph-1] && ml[ph] >= ml[ph-1]);
    int phtl = tupLengths[ph-1];
    currLength = extnLengths[ph-1];
    if (VERBOSE) cout << "Phase tuple length: " << phtl << "; extension length: " << currLength << endl;
    
    breakCnt[ph-1] = breakptFind(htuples[ph-1], hidx[ph-1], htupImg[ph-1], hl[ph-1],
				 mtuples[ph-1], midx[ph-1], mtupImg[ph-1], ml[ph-1], 
				 phtl, hbreaks[ph-1], mbreaks[ph-1]);
    if (ALIGNVERBOSE) cout << endl << "found " << breakCnt[ph-1] << " " << phtl << "-tuple breakpoints" << endl;
    
    int *breaksValid = new int[breakCnt[ph-1]]; arrayZero(breaksValid, breakCnt[ph-1]);
    int accepted=0, rejected=0;
    for (i=0; i<breakCnt[ph-1]; ++i) {
      if (ALIGNVERBOSE)       cout << "Constr# " << i << "\t" << hbreaks[ph-1][i] << " ==>> " << mbreaks[ph-1][i] << "\t";
      if (i && (hbreaks[ph-1][i] - hbreaks[ph-1][i-1] < phtl || mbreaks[ph-1][i] - mbreaks[ph-1][i-1] < phtl)) // Assertions;
	assert(mbreaks[ph-1][i] - mbreaks[ph-1][i-1] == hbreaks[ph-1][i] - hbreaks[ph-1][i-1]);

      if (himgFixed[hbreaks[ph-1][i]] == mbreaks[ph-1][i] || himgFixed[hbreaks[ph-1][i] + phtl - 1] == mbreaks[ph-1][i] + phtl - 1) {
	if (ALIGNVERBOSE) cout << "Fixed" << endl;
	breaksValid[i] = 1;
	continue;
      }

      if (hbreaks[ph-1][i] <= extnLengths[ph-1] || hbreaks[ph-1][i] + phtl >= hseql-extnLengths[ph-1] ||
	  mbreaks[ph-1][i] <= extnLengths[ph-1] || mbreaks[ph-1][i] + phtl >= mseql-extnLengths[ph-1]) {
	breaksValid[i] = 1;
	if (ALIGNVERBOSE) cout << "Accepted" << endl;
	continue;
      }
      int    *himageLeft = new int[currLength], *himageRight = new int[currLength];
      int    *mimageLeft = new int[currLength], *mimageRight = new int[currLength];
      int disagreeWithFixedLeft=0, disagreeWithFixedRight=0;
      
      arrayZero(himageLeft,  currLength); arrayZero(himageRight, currLength);
      arrayZero(mimageLeft,  currLength); arrayZero(mimageRight, currLength);
      
      assert(hbreaks[ph-1][i]+phtl+currLength-1 <= hseql && hbreaks[ph-1][i]-currLength >= 0);
      assert(mbreaks[ph-1][i]+phtl+currLength-1 <= mseql && mbreaks[ph-1][i]-currLength >= 0);
	
      alignScoreLeft  = gappedAlign(hseqInt, mseqInt,
				    hbreaks[ph-1][i] - currLength, hbreaks[ph-1][i] - 1,
				    mbreaks[ph-1][i] - currLength, mbreaks[ph-1][i] - 1,
				    1, -1, -2, -1,
				    himageLeft, mimageLeft);
      for (j=currLength; j>0; --j)
	if (himgFixed[hbreaks[ph-1][i] - j] > 0 &&
	    (himageLeft[currLength - j] == -1 || himgFixed[hbreaks[ph-1][i] - j] != mbreaks[ph-1][i] - currLength + himageLeft[currLength - j]))
	  disagreeWithFixedLeft = 1;

      alignScoreRight = gappedAlign(hseqInt, mseqInt,
				    hbreaks[ph-1][i] + phtl, hbreaks[ph-1][i] + phtl + currLength-1,
				    mbreaks[ph-1][i] + phtl, mbreaks[ph-1][i] + phtl + currLength-1,
				    1, -1, -2, -1,
				    himageRight, mimageRight);
      for (j=0; j<currLength; ++j)
	if (himgFixed[hbreaks[ph-1][i] + phtl + j] > 0 &&
	    (himageRight[j] == -1 || himgFixed[hbreaks[ph-1][i] + phtl + j] != mbreaks[ph-1][i] + phtl + himageLeft[j]))
	  disagreeWithFixedRight = 1;

      if (alignScoreLeft + alignScoreRight >= extnCutoffs[ph-1] && !disagreeWithFixedLeft && !disagreeWithFixedRight) {
	breaksValid[i] = 1;
	
	if (ALIGNVERBOSE) {
	  cout << "Accepted, scores: " << alignScoreLeft << "," << alignScoreRight;
	  if (disagreeWithFixedLeft) cout << "\t oops L";
	  if (disagreeWithFixedRight) cout << "\t oops R";
	  if (!disagreeWithFixedLeft && !disagreeWithFixedRight) cout << "\t OK";
	  cout << endl;
	}
	accepted++;
      }
      else {
	if (ALIGNVERBOSE) cout << "Rejected, scores: " << alignScoreLeft << "," << alignScoreRight << " \t disagreements: " 
			       << disagreeWithFixedLeft << "," << disagreeWithFixedRight << endl;
	rejected++;
      }
      delete[] himageLeft; delete[] himageRight;
      delete[] mimageLeft; delete[] mimageRight;
    }
    if (ALIGNVERBOSE)     cout << "Accepted: " << accepted << "; Rejected: " << rejected << endl;
    
    arrayInit(-1,hconstrExplicit, hseql+1);
    for (i=0; i<breakCnt[ph-1]; ++i)
      if (breaksValid[i]) {
	for (j=hbreaks[ph-1][i]; j < hbreaks[ph-1][i] + phtl; ++j) {
	  himgFixed[j] = mbreaks[ph-1][i] + (j-hbreaks[ph-1][i]);
	  mimgFixed[mbreaks[ph-1][i] + (j-hbreaks[ph-1][i])] = j;
	}
	for (j=0; j <= tupLengths[ph-1]-tupLengths[ph]; ++j) {
	  if (ALIGNVERBOSE && hconstrExplicit[hbreaks[ph-1][i] + j] == -1)
	    cout << "  constr# " << i << " sets " << hbreaks[ph-1][i]+j << " > " << mbreaks[ph-1][i]+j << endl;
	  hconstrExplicit[hbreaks[ph-1][i]+j] = mbreaks[ph-1][i]+j;
	}
      }
    delete[] breaksValid;

    lastConstrPos = -10000;
    for (i=2; i<hseql+1; ++i) {
      if (lastConstrPos > i - tupLengths[ph])
	if (hconstrExplicit[i]==-1 && hconstrExplicit[i+1]>=0 &&
	    i+1 - lastConstrPos == hconstrExplicit[i+1] - hconstrExplicit[lastConstrPos])
	  for (j=1; j+lastConstrPos <= i; ++j) {
	    if (ALIGNVERBOSE && hconstrExplicit[j+lastConstrPos] == -1)
	      cout << "    post-processing sets " << j+lastConstrPos 
		   << " > " << hconstrExplicit[lastConstrPos] + j << endl;
	    hconstrExplicit[j+lastConstrPos] = hconstrExplicit[lastConstrPos] + j;
	  }
      if (hconstrExplicit[i]>=0) lastConstrPos = i;
    }

    arrayInit(-1, hinverseIdx, hseql+1); for (i=0; i<hl[ph]; ++i) hinverseIdx[hidx[ph][i]] = i;
    arrayInit(-1, minverseIdx, mseql+1); for (i=0; i<ml[ph]; ++i) minverseIdx[midx[ph][i]] = i;
    
    arrayZero(hconstraints, hseql+1);
    arrayZero(mconstraints, mseql+1);
    
    int constraintCnt = 0;
    
    for (i=1; i<=hseql; ++i)
      if (hconstrExplicit[i] >=0) {
	assert(hinverseIdx[i]>=0 && minverseIdx[hconstrExplicit[i]]>=0);
	hconstraints[constraintCnt] = hinverseIdx[i];
	mconstraints[constraintCnt] = minverseIdx[hconstrExplicit[i]];
	constraintCnt++;
      }
    for (i=0; i<constraintCnt; ++i)
      if (ALIGNVERBOSE) cout << "Accepted Constr# " << i << "\t" << hconstraints[i] << " ==> " << mconstraints[i] 
	   << "  i.e. " << hidx[ph][hconstraints[i]] << " => " << midx[ph][mconstraints[i]] << endl;
   if (ALIGNVERBOSE)  cout << " constraint count: " << constraintCnt << endl;
    
   constraintsalignUseExtensions(htuples[ph], mtuples[ph], hl[ph], ml[ph], 1, 0, 0, htupImg[ph], mtupImg[ph],
				 hconstraints, mconstraints, constraintCnt, 1,
				 hseqInt, mseqInt, hidx[ph], midx[ph], hseql, mseql, extnLengths[ph], tupLengths[ph]);
  }  
  ph = phaseCount-1;
  breakCnt[ph] = breakptFind(htuples[ph], hidx[ph], htupImg[ph], hl[ph],
			     mtuples[ph], midx[ph], mtupImg[ph], ml[ph], tupLengths[ph], hbreaks[ph], mbreaks[ph]);

  int phtl = tupLengths[ph];
  int *breaksValid = new int[breakCnt[ph]]; arrayZero(breaksValid, breakCnt[ph]);
  int accepted=0, rejected=0;
  for (i=0; i<breakCnt[ph]; ++i) {
    if (ALIGNVERBOSE)       cout << "Constr# " << i << "\t" << hbreaks[ph][i] << " ==>> " << mbreaks[ph][i] << "\t";
    if (hbreaks[ph][i] <= extnLengths[ph] || hbreaks[ph][i] + phtl >= hseql-extnLengths[ph] ||
	mbreaks[ph][i] <= extnLengths[ph] || mbreaks[ph][i] + phtl >= mseql-extnLengths[ph]) {
      breaksValid[i] = 1;
      if (ALIGNVERBOSE) cout << endl;
      continue;
    }
    currLength = extnLengths[ph];
    int    *himageLeft = new int[currLength], *himageRight = new int[currLength];
    int    *mimageLeft = new int[currLength], *mimageRight = new int[currLength];
    int disagreeWithFixedLeft=0, disagreeWithFixedRight=0;

    arrayZero(himageLeft,  currLength); arrayZero(himageRight, currLength);
    arrayZero(mimageLeft,  currLength); arrayZero(mimageRight, currLength);

    assert(hbreaks[ph][i]+phtl+currLength-1 <= hseql && hbreaks[ph][i]-currLength >= 0);
    assert(mbreaks[ph][i]+phtl+currLength-1 <= mseql && mbreaks[ph][i]-currLength >= 0);
    
    alignScoreLeft  = gappedAlign(hseqInt, mseqInt,
				  hbreaks[ph][i] - currLength, hbreaks[ph][i] - 1,
				  mbreaks[ph][i] - currLength, mbreaks[ph][i] - 1,
				  1, -1, -2, -1,
				  himageLeft, mimageLeft);
    
    for (j=currLength; j>0; --j)
      if (himgFixed[hbreaks[ph][i] - j] > 0 &&
	  (himageLeft[currLength - j] == -1 || himgFixed[hbreaks[ph][i] - j] != mbreaks[ph][i] - currLength + himageLeft[currLength - j]))
	disagreeWithFixedLeft = 1;
    
    alignScoreRight = gappedAlign(hseqInt, mseqInt,
				  hbreaks[ph][i] + phtl, hbreaks[ph][i] + phtl + currLength-1,
				  mbreaks[ph][i] + phtl, mbreaks[ph][i] + phtl + currLength-1,
				  1, -1, -2, -1,
				  himageRight, mimageRight);
    
    for (j=0; j<currLength; ++j)
      if (himgFixed[hbreaks[ph][i] + phtl + j] > 0 &&
	  (himageRight[j] == -1 || himgFixed[hbreaks[ph][i] + phtl + j] != mbreaks[ph][i] + phtl + himageLeft[j]))
	disagreeWithFixedRight = 1;
    
    /*
    cout << endl << "\t\t";
    for (j=0; j<currLength; ++j) cout << hbreaks[ph-1][i] + phtl + j << ": " << mbreaks[ph-1][i] + phtl + himageLeft[j] << "    ";
    cout << endl;
    */

    if (alignScoreLeft + alignScoreRight >= extnCutoffs[ph] && !disagreeWithFixedLeft && !disagreeWithFixedRight) {
      breaksValid[i] = 1;
      if (ALIGNVERBOSE) cout << "Accepted, scores: " << alignScoreLeft << "," << alignScoreRight << endl;
      accepted++;
    }
    else {
      if (ALIGNVERBOSE) cout << "Rejected, scores: " << alignScoreLeft << "," << alignScoreRight << endl;
      rejected++;
    }
    delete[] himageLeft; delete[] himageRight;
    delete[] mimageLeft; delete[] mimageRight;
  }
  if (ALIGNVERBOSE)     cout << "Accepted: " << accepted << "; Rejected: " << rejected << endl;
  
  arrayInit(-1,hconstrExplicit, hseql+1);
  for (i=0; i<breakCnt[ph]; ++i)
    if (breaksValid[i])
      for (j=0; j <= tupLengths[ph]-1; ++j) {
	if (ALIGNVERBOSE && hconstrExplicit[hbreaks[ph][i] + j] == -1)
	  cout << "  constr# " << i << " sets " << hbreaks[ph][i]+j << " > " << mbreaks[ph][i]+j << endl;
	hconstrExplicit[hbreaks[ph][i]+j] = mbreaks[ph][i]+j;
      }
  delete[] breaksValid;
  
  arrayZero(hconstraints, hseql+1);
  arrayZero(mconstraints, mseql+1);
  
  int constraintCnt = 0;
  
  for (i=1; i<=hseql; ++i)
    if (hconstrExplicit[i] >=0) {
      hconstraints[constraintCnt] = i;                         humImg[i]           = hconstrExplicit[i];
      mconstraints[constraintCnt] = hconstrExplicit[i];        mouseImg[humImg[i]] = i;
      constraintCnt++;
    }
  for (i=0; i<constraintCnt; ++i)
    if (ALIGNVERBOSE) cout << "Accepted Constr# " << i << "\t" << hconstraints[i] << " ==> " << mconstraints[i] 
			   << "  i.e. " << hidx[ph][hconstraints[i]] << " => " << midx[ph][mconstraints[i]] << endl;
  if (ALIGNVERBOSE)  cout << " constraint count: " << constraintCnt << endl;
  
  constrainedAlign(hseq, mseq, hconstraints, mconstraints, constraintCnt, 1, 1, -1, -2, humImg, mouseImg, 7, 0);
  
  for (i=1; i<=hseql; ++i) if (humImg[i] > 0) assert(humImg[i] <= mseql && mouseImg[humImg[i]] == i);
  
  for (i=0; i<phaseCount; ++i) {
    delete[] htuples[i]; delete[] hidx[i]; delete[] htupImg[i]; delete[] hbreaks[i];
    delete[] mtuples[i]; delete[] midx[i]; delete[] mtupImg[i]; delete[] mbreaks[i];
  }
  delete[] htuples; delete[] hidx; delete[] htupImg; delete[] hbreaks;
  delete[] mtuples; delete[] midx; delete[] mtupImg; delete[] mbreaks;
  
  delete[] hl; delete[] ml;
  delete[] himgFixed;
  delete[] mimgFixed;
  
  delete[] breakCnt;
  delete[] hconstrExplicit;
  delete[] hinverseIdx;
  delete[] minverseIdx;
  delete[] hconstraints;
  delete[] mconstraints;
  
  return TRUE;
}

int phaseAlign(FilterSequence *hseq, FilterSequence *mseq, int phaseCount, int *tupLengths, int *humImg, int *mouseImg) {
  assert(phaseCount>0);

  assert(0); // We are not using this phase align;
  int ph = 0, i=0,j=0, hseql = hseq->get_length(), mseql = mseq->get_length();
  
   for (i=0; i<phaseCount; ++i) {
     assert(tupLengths[i]>0);
     if (i) assert(tupLengths[i] < tupLengths[i-1]);
   }

   int **htuples = new (int*)[phaseCount], **hidx = new (int*)[phaseCount];
   int **mtuples = new (int*)[phaseCount], **midx = new (int*)[phaseCount];
   for (i=0; i<phaseCount; ++i) {
     htuples[i] = new int[hseql+1]; arrayZero(htuples[i], hseql+1);
     mtuples[i] = new int[mseql+1]; arrayZero(mtuples[i], mseql+1);
     hidx[i]    = new int[hseql+1]; arrayZero(hidx[i],    hseql+1);
     midx[i]    = new int[mseql+1]; arrayZero(midx[i],    mseql+1);
   }
   int *hl = new int[phaseCount]; arrayZero(hl, phaseCount);
   int *ml = new int[phaseCount]; arrayZero(ml, phaseCount);
   
   for (i=0; i<phaseCount; ++i) {
     exactMatchFind(hseq, mseq, tupLengths[i], htuples[i], hidx[i], hl[i], mtuples[i], midx[i], ml[i]);
     if (ALIGNVERBOSE) cout << "Found " << hl[i]  << " , " << ml[i]  << " " << tupLengths[i]  << "-tuple matches " << endl;
   }
   int **htupImg = new (int*)[phaseCount], **hbreaks = new (int*)[phaseCount];
   int **mtupImg = new (int*)[phaseCount], **mbreaks = new (int*)[phaseCount];
   for (i=0; i<phaseCount; ++i) {
     htupImg[i] = new int[ hl[i] ]; arrayInit(-1, htupImg[i], hl[i]);
     hbreaks[i] = new int[ hl[i] ]; arrayInit(-1, hbreaks[i], hl[i]);
     mtupImg[i] = new int[ ml[i] ]; arrayZero(mtupImg[i], ml[i]);
     mbreaks[i] = new int[ ml[i] ]; arrayZero(mbreaks[i], ml[i]);
   }
   int *breakCnt = new int[phaseCount]; arrayZero(breakCnt, phaseCount);
   int *hconstrExplicit = new int[hseql+1];
   int *hinverseIdx = new int[hseql+1];
   int *minverseIdx = new int[mseql+1];
   int *hconstraints = new int[hseql+1];
   int *mconstraints = new int[mseql+1];

   int firstPass = 0;
   double firstPassScore = 0;

   for (ph = 0; ph<phaseCount; ++ph) {
     if (!firstPass) {
       if (ALIGNVERBOSE) cout << " trying " << tupLengths[ph] << "-tuples " << endl;
       firstPassScore = memalign(htuples[ph], mtuples[ph], 0, hl[ph]-1, 0, ml[ph]-1, 1, -1, -1, htupImg[ph], mtupImg[ph]);
       firstPassScore = tupleAlignmentScore(htuples[ph], mtuples[ph], 0, hl[ph]-1, 0, ml[ph]-1, 1, -1, -1, 0.8, 1.5, htupImg[ph], mtupImg[ph]);
       if (firstPassScore < 0) continue;
       firstPass = 1;
       if (ALIGNVERBOSE) cout << "first pass score: " << firstPassScore << "     tuple size: " << tupLengths[ph] << endl;
       continue;
     }
     assert(hl[ph] >= hl[ph-1] && ml[ph] >= ml[ph-1]);

     breakCnt[ph-1] = breakptFind(htuples[ph-1], hidx[ph-1], htupImg[ph-1], hl[ph-1],
				  mtuples[ph-1], midx[ph-1], mtupImg[ph-1], ml[ph-1], tupLengths[ph-1], hbreaks[ph-1], mbreaks[ph-1]);
     if (ALIGNVERBOSE) cout << "found " << breakCnt[ph-1] << " " << tupLengths[ph-1] << "-tuple breakpoints" << endl;
     for (i=1; i<breakCnt[ph-1]; ++i) {
       if (ALIGNVERBOSE) cout << " long " << hbreaks[ph-1][i] << " ==>> " << mbreaks[ph-1][i] << endl;
       if (hbreaks[ph-1][i] - hbreaks[ph-1][i-1] < tupLengths[ph-1] || mbreaks[ph-1][i] - mbreaks[ph-1][i-1] < tupLengths[ph-1])    // Assertions;
	 assert(mbreaks[ph-1][i] - mbreaks[ph-1][i-1] == hbreaks[ph-1][i] - hbreaks[ph-1][i-1]);
     }  
     


     arrayInit(-1,hconstrExplicit, hseql+1);
     for (i=0; i<breakCnt[ph-1]; ++i)
       for (j=0; j <= tupLengths[ph-1]-tupLengths[ph]; ++j) 
	 hconstrExplicit[hbreaks[ph-1][i]+j] = mbreaks[ph-1][i]+j;
     
     arrayInit(-1, hinverseIdx, hseql+1); for (i=0; i<hl[ph]; ++i) hinverseIdx[hidx[ph][i]] = i;
     arrayInit(-1, minverseIdx, mseql+1); for (i=0; i<ml[ph]; ++i) minverseIdx[midx[ph][i]] = i;
     
    arrayZero(hconstraints, hseql+1);
    arrayZero(mconstraints, mseql+1);

    int constraintCnt = 0;

    for (i=1; i<=hseql; ++i)
      if (hconstrExplicit[i] >=0) {
	assert(hinverseIdx[i]>=0 && minverseIdx[hconstrExplicit[i]]>=0);
	hconstraints[constraintCnt] = hinverseIdx[i];
	mconstraints[constraintCnt] = minverseIdx[hconstrExplicit[i]];
	constraintCnt++;
      }
    if (ALIGNVERBOSE) cout << " constraint count: " << constraintCnt << endl;

    constraintsalign(htuples[ph], mtuples[ph], hl[ph], ml[ph], 1, -1, -1, htupImg[ph], mtupImg[ph],
		     hconstraints, mconstraints, constraintCnt, 0);
   }
   
   ph = phaseCount-1;
   breakCnt[ph] = breakptFind(htuples[ph], hidx[ph], htupImg[ph], hl[ph],
			      mtuples[ph], midx[ph], mtupImg[ph], ml[ph], tupLengths[ph], hbreaks[ph], mbreaks[ph]);
    
   constrainedAlign(hseq, mseq, hbreaks[ph],mbreaks[ph], breakCnt[ph], tupLengths[ph], 1, -1, -3, humImg, mouseImg);

   for (i=1; i<=hseql; ++i) if (humImg[i] > 0) assert(humImg[i] <= mseql && mouseImg[humImg[i]] == i);
   
   for (i=0; i<phaseCount; ++i) {
     delete[] htuples[i]; delete[] hidx[i]; delete[] htupImg[i]; delete[] hbreaks[i];
     delete[] mtuples[i]; delete[] midx[i]; delete[] mtupImg[i]; delete[] mbreaks[i];
   }
   delete[] htuples; delete[] hidx; delete[] htupImg; delete[] hbreaks;
   delete[] mtuples; delete[] midx; delete[] mtupImg; delete[] mbreaks;

   delete[] hl; delete[] ml;

   delete[] breakCnt;
   delete[] hconstrExplicit;
   delete[] hinverseIdx;
   delete[] minverseIdx;
   delete[] hconstraints;
   delete[] mconstraints;

   return TRUE;
}



double tupleAlignmentScore(int *htup, int *mtup, int hfrom, int hto, int mfrom, int mto,
			   double match, double mismatch, double gap, double alpha, double beta,
			   int *himg, int *mimg) {
  int i=0;
  double score=0;

  int matchCount = 0;
  for (i=hfrom; i<=hto; ++i)
    if (himg[i]>=0)
      if (htup[i] == mtup[himg[i]]) {
	score += match * power(beta, matchCount);
	matchCount++;
      }
      else {
	score += mismatch;
	matchCount = 0;
      }

  int gapCount = 0;
  for (i=hfrom; i<=hto; ++i)
    if (himg[i]>=0) gapCount=0;
    else {
      assert(himg[i]==-1);
      score += gap * power(alpha, gapCount);
      gapCount++;
    }
  gapCount = 0;
  for (i=mfrom; i<=mto; ++i)
    if (mimg[i]>=0) gapCount=0;
    else {
      assert(mimg[i]==-1);
      score += gap * power(alpha, gapCount);
      gapCount++;
    }

  return score;
}

double exonAlignmentScore(FilterSequence *hseq, FilterSequence *mseq, int hfrom, int hto, int mfrom, int mto,
			  double match, double mismatch, double gap, double alpha, double beta, int cap,
			  int *himg, int *mimg, int isATG) {
  int i=0;
  double score = 0;
  double score1 =0;
  double score2 = 0;
  double score3 = 0;
  int matchCount = 0, gapCount = 0;
  
  int *cumHGaps = new int[hto-hfrom+1]; arrayZero(cumHGaps,hto-hfrom+1);
  int *cumMGaps = new int[mto-mfrom+1]; arrayZero(cumMGaps,mto-mfrom+1);

  int lastTotal = 0;
  for (i=0; i<hto-hfrom+1; ++i)
    if (himg[hfrom+i] == -1 || himg[hfrom+i] < mfrom || himg[hfrom+i] > mto) 
      cumHGaps[i] = ++lastTotal;
    else 
      cumHGaps[i] =   lastTotal;

  lastTotal = 0;
  for (i=0; i<mto-mfrom+1; ++i)
    if (mimg[mfrom+i] == -1 || mimg[mfrom+i] < hfrom || mimg[mfrom+i] > hto)
      cumMGaps[i] = ++lastTotal;
    else
      cumMGaps[i] =   lastTotal;

  for (i=hfrom; i<hto; ++i)
    if (himg[i] >= 0) {
      assert(himg[i]);
      gapCount = 0;
      if (hseq->get(i) == mseq->get(himg[i])) {
	if (himg[i] < mfrom || himg[i] > mto) continue;
	if (cumHGaps[i-hfrom] % 3     == cumMGaps[himg[i]-mfrom] % 3) score1 += match * power(beta, matchCount);
	else                                                          score1 -= match/2;
	if ((cumHGaps[i-hfrom]+1) % 3 == cumMGaps[himg[i]-mfrom] % 3) score2 += match * power(beta, matchCount);
	else                                                          score2 -= match/2;
	if ((cumHGaps[i-hfrom]+2) % 3 == cumMGaps[himg[i]-mfrom] % 3) score3 += match * power(beta, matchCount);
	else                                                          score3 -= match/2;
	if (matchCount<cap) matchCount++;
      }
      else {
	matchCount = 0;
	score += mismatch;
      }
    }
    else {
      matchCount = 0;
      score += gap * power(alpha, MIN(gapCount, 5)) * power((4.0+alpha)/5.0, max(0,gapCount-8));
      gapCount++;
    }
  gapCount = 0;
  for (i=mfrom; i<mto; ++i)
    if (mimg[i] >= 0) { assert(mimg[i]); gapCount = 0; }
    else {
      score += gap * power(alpha, gapCount);
      if (gapCount<cap) gapCount++;
    }
  
  score+=isATG ? MAX(score1,score2,score3) : score1;

  delete[] cumHGaps;
  delete[] cumMGaps;
  
  return score;
}




int extractAligningRegions(int *hseqInt, int *mseqInt, int *himg, int *mimg, int hseql, int mseql,
			   int maxgap, int minreglen,
			   int *hbegins, int *hends, int *mbegins, int *mends) {
  
  int i,j;
  
  int *hcumGaps = new int[hseql+1]; arrayZero(hcumGaps, hseql+1);
  int *mcumGaps = new int[mseql+1]; arrayZero(mcumGaps, mseql+1);

  int MASKED = 7777777;

  int *hcumMatches = new int[hseql+1]; arrayZero(hcumGaps, hseql+1);
  int *mcumMatches = new int[mseql+1]; arrayZero(mcumGaps, mseql+1);
  
  
  for (i=1; i<=hseql; ++i) 
    if (himg[i] != -1 && hseqInt[i] == mseqInt[himg[i]]) hcumMatches[i] = hcumMatches[i-1] + 1;
    else                                                 hcumMatches[i] = hcumMatches[i-1];
  
  for (i=1; i<=mseql; ++i) 
    if (mimg[i] != -1 && mseqInt[i] == hseqInt[mimg[i]]) mcumMatches[i] = mcumMatches[i-1] + 1;
    else                                                 mcumMatches[i] = mcumMatches[i-1];

  
  int windowLength = (int)(1.5*minreglen);
  int leastMatches = (int)(0.27*windowLength);

  
  for (i=1; i<hseql-windowLength; ++i)
    if (hcumMatches[i+windowLength] - hcumMatches[i] < leastMatches) hcumGaps[i+windowLength/2] = MASKED;
  for (i=1; i<mseql-windowLength; ++i)
    if (mcumMatches[i+windowLength] - mcumMatches[i] < leastMatches) mcumGaps[i+windowLength/2] = MASKED;
 
  for (i=1; i<=hseql; ++i) if (himg[i] == -1) hcumGaps[i] = hcumGaps[i-1] + 1;
  for (i=1; i<=mseql; ++i) if (mimg[i] == -1) mcumGaps[i] = mcumGaps[i-1] + 1;
  
  // Step 1:                       Mask all gaps;
  int last = hcumGaps[hseql-1];
  for (i=hseql-1; i>=0; --i)
    if (hcumGaps[i]) {
      if (last) hcumGaps[i] = last;
      else last = hcumGaps[i];
      if (hcumGaps[i] >= maxgap) hcumGaps[i] = MASKED;
    }
    else last = 0;
  last = 0;
  for (i=mseql; i>=0; --i)
    if (mcumGaps[i]) {
      if (last) mcumGaps[i] = last;
      else last = mcumGaps[i];
      if (mcumGaps[i] >= maxgap) mcumGaps[i] = MASKED;
    }
    else last = 0;

  
  int changed = 0; cout << endl;
  
  do {
    // Step 2:                        Mask short regions;
    last = 0;
    
    for (i=1; i<=hseql; ++i)
      if (hcumGaps[i] == MASKED) {
	if (last < i-1 && i-last-1 < minreglen) for (j=MAX(0,last+1); j<i; ++j) hcumGaps[j] = MASKED;
	last = i;
      }
    last = 0;
    for (i=1; i<mseql; ++i)
      if (mcumGaps[i] == MASKED) {
	if (last < i-1 && i-last-1 < minreglen) for (j=MAX(0,last+1); j<i; ++j) mcumGaps[j] = MASKED;
	last = i;
      }
    
    // Step 3:                        Mask regions cross-species;
    changed = 0;
    for (i=1; i<=hseql; ++i)
      if (hcumGaps[i] == MASKED && himg[i] != -1) {
	assert(himg[i] >=0 && himg[i] <= mseql);
	if (mcumGaps[himg[i]] != MASKED) {
	  mcumGaps[himg[i]] = MASKED;
	  changed++;
	}
      }
      else if (himg[i] != -1) {
	assert(himg[i] >= 0 && himg[i] <= mseql);
	if (mcumGaps[himg[i]] == MASKED)
	  if (hcumGaps[i] != MASKED) {
	    hcumGaps[i] = MASKED;
	    changed++;
	  }
      }
    
    if (hcumGaps[1])      hcumGaps[1]     = MASKED;
    if (mcumGaps[1])      mcumGaps[1]     = MASKED;
    if (hcumGaps[hseql])  hcumGaps[hseql] = MASKED;
    if (mcumGaps[mseql])  mcumGaps[mseql] = MASKED;
    for (i=2; i<=hseql; ++i)   if (hcumGaps[i-1] == MASKED && himg[i] == -1) hcumGaps[i] = MASKED;
    for (i=2; i<=mseql; ++i)   if (mcumGaps[i-1] == MASKED && mimg[i] == -1) mcumGaps[i] = MASKED;
    for (i=hseql-1; i>=1; --i) if (hcumGaps[i+1] == MASKED && himg[i] == -1) hcumGaps[i] = MASKED;
    for (i=mseql-1; i>=1; --i) if (mcumGaps[i+1] == MASKED && mimg[i] == -1) mcumGaps[i] = MASKED;
  } while (changed);
  
  int hregnum = 0, mregnum = 0;
  int inRegion = 0;
  
  for (i=1; i<=hseql; ++i)
    if (hcumGaps[i] != MASKED) {
      if (!inRegion && i < hseql-minreglen) {
	inRegion = 1;
	hregnum++;
      }
    }
    else {
      inRegion = 0;
    }
  inRegion = 0;
  for (i=1; i<=mseql; ++i)
    if (mcumGaps[i] != MASKED) {
      if (!inRegion && i < mseql - minreglen) {
	inRegion = 1;
	mregnum++;
      }
    }
    else inRegion = 0;
  

  int *hbegins_aux = new int[hregnum+2];  arrayInit(-1, hbegins_aux, hregnum+2);
  int *hends_aux   = new int[hregnum+2];  arrayInit(-1, hends_aux,   hregnum+2);
  int *mbegins_aux = new int[mregnum+2];  arrayInit(-1, mbegins_aux, mregnum+2);
  int *mends_aux   = new int[mregnum+2];  arrayInit(-1, mends_aux,   mregnum+2);


  
  int regptr = 0; inRegion = 0;
  for (i=1; i<=hseql; ++i)
    if (!inRegion && hcumGaps[i] != MASKED && i < hseql-minreglen && (i==1 || hcumGaps[i-1] == MASKED)) { 
      hbegins_aux[regptr] = i;
      inRegion = 1;
      assert(himg[i] != -1); }
    else if (inRegion && hcumGaps[i] != MASKED && (i==hseql || hcumGaps[i+1] == MASKED)) { 
      hends_aux[regptr++] = i;
      inRegion = 0;
      assert(himg[i] != -1); }
  
  assert(regptr==hregnum);
  
  regptr = inRegion = 0;
  for (i=1; i<=mseql; ++i)
    if (!inRegion && mcumGaps[i] != MASKED && i < mseql-minreglen && (i==1 || mcumGaps[i-1] == MASKED)) { 
      mbegins_aux[regptr] = i;
      inRegion = 1;
      assert(mimg[i] != -1);
    }
    else
      if (inRegion && mcumGaps[i] != MASKED && (i==mseql || mcumGaps[i+1] == MASKED)) { 
	mends_aux[regptr++] = i;
	inRegion = 0;
	assert(mimg[i] != -1);
      }
  assert(regptr==mregnum);
  
  int *hendpts = new int[hseql+1]; arrayZero(hendpts,hseql+1);

  for (i=0; i<hregnum; ++i) {
    hendpts[hbegins_aux[i]] = 1;
    hendpts[hends_aux[i]]   = 2;
  }
  for (i=0; i<mregnum; ++i) {
    hendpts[mimg[mbegins_aux[i]]] = 1;
    hendpts[mimg[mends_aux[i]]]   = 2;
  }
  int regnum = 0;

  last = 0; inRegion = 0;
  for (i=0; i<=hseql; ++i) {
    if (hendpts[i] == 1) {
      last = i;
      inRegion = 1;
    }
    if (hendpts[i] == 2) {
      if (i - last >= minreglen && inRegion) {
	hbegins[regnum] = last;       hends[regnum] = i;
	mbegins[regnum] = himg[last]; mends[regnum] = himg[i];
	if (ALIGNVERBOSE) cout << "Region accepted: " << last << "," << i << " -> " << himg[last] << "," << himg[i-1] << endl;
	regnum++;
      }
      inRegion = 0;
    }
  }

  
  delete[] hendpts;
  delete[] hbegins_aux;
  delete[] hends_aux;
  delete[] mbegins_aux;
  delete[] mends_aux;
  delete[] hcumGaps;
  delete[] mcumGaps;
  
  return regnum;
}


void outputAlignment(ofstream &fout, FilterSequence *hseq, FilterSequence *mseq, int *himg, int *mimg) {
  int p1 = 1, p2 = 1;
  int hseql = hseq->get_length();
  int mseql = mseq->get_length();

  while (1) {
    if (p1 > hseql || p2 > mseql) break;
    if (himg[p1] > 0 && mimg[p2] > 0)  {
      assert(himg[p1] == p2 && mimg[p2] == p1);
      
      fout << p1 << "  " << nucl2char(hseq->get(p1));
      if (hseq->get(p1) == mseq->get(p2)) fout << " - ";
      else                                     fout << "   ";
      fout << nucl2char(mseq->get(p2)) << "  " << p2 << endl;
      p1++; p2++;
    }
    else if (himg[p1] < 0 &&  mimg[p2] > 0) {
      fout << p1 << "  " << nucl2char(hseq->get(p1)) << "   |   " << endl;
      p1++;
    }
    else if (himg[p1] > 0 &&  mimg[p2] < 0) {
      fout << "     |   " << nucl2char(mseq->get(p2)) << "  " << p2 << endl;
      p2++;
    }
    if (p1 <= hseql && p2 <= mseql)
      assert(himg[p1] >=0 || mimg[p2] >= 0);
  }
  fout << endl << endl;
}

double pamscore(int aa1, int aa2) {
  assert(aa1>=0 && aa1 <= 21 && aa2 >= 0 && aa2 <= 21);
  return currAlignPAM[aaOrder[aa1]][aaOrder[aa2]];
}

double findmapPAM(int *seq1, int *seq2, int from1, int to1, int from2, int to2,  
		  int matrixIdx, int *imagefirst, int *imagesecond, double *col1, double *col2, double *prebest, 
		  double *suffbest) {
  
  int i,j, k;
  int lengthfirst = to1 - from1 + 1;
  int lengthsecond = to2 - from2 + 1;
  if(lengthfirst==0 || lengthsecond == 0) return 0;
  int m=(int)ceil(((double)lengthfirst)/2)-1+from1;

  int mu = m-from1+1;
  double scorek[2]; int maxindex=0; int gapflag=0; int tempgapflag=0; double maxscore=-Infinity;
  double up, left, upleft;
  
  double gap = pamscore(21, 1);

  k = lengthsecond;
  // Prefix Calculation
  // Boundary Conditions:
  prebest[0] = gap*(m-from1);
  
  for (i=0; i<=mu-1; ++i) col1[i]=gap*i;

  for (j=1; j<=k; ++j){
    col2[0]=gap*j;
    for (i=1; i<=mu-1; ++i) {
      up=col2[i-1]+gap;
      left=col1[i]+gap;
      upleft = col1[i-1] + pamscore(seq1[from1+i-1], seq2[from2+j-1]);
      col2[i]=MAX(up,left,upleft);
      col1[i-1]=col2[i-1];
      if (i==mu-1 && j!=k)
	col1[mu-1]=col2[mu-1];
    }
    prebest[j]=col2[mu-1]; 
  }
  if (m==from1)
    for (j=1; j<=k; ++j)
      prebest[j]=gap*j;
  
  
  k=0;
  // Suffix Calculation
  // Boundary Conditions:
  suffbest[0]=gap*(to1-m);
  for (i=0; i<=to1-m; ++i)
    col1[i]=gap*i;
  for (j=1; j<=lengthsecond-k; ++j){
    col2[0]=gap*j;
    for (i=1; i<=to1-m; ++i){
      up=col2[i-1]+gap;
      left=col1[i]+gap;
      upleft=col1[i-1] + pamscore(seq1[to1-i+1], seq2[to2-j+1]);
      col2[i]=MAX(up,left,upleft);
      col1[i-1]=col2[i-1];
      if (i==to1-m && j!=lengthsecond-k)
	col1[to1-m]=col2[to1-m];
    }
    suffbest[j]=col2[to1-m];
  }
  if (m==to1)
    for (j=1; j<=lengthsecond; ++j)
      suffbest[j]=gap*j;
  
  for (k=0; k<=lengthsecond; ++k){
    if (k>0){
      scorek[0]=prebest[k-1]+suffbest[lengthsecond-k]+pamscore(seq1[m], seq2[k+from2-1]);
      scorek[1]=prebest[k]+suffbest[lengthsecond-k]+gap;
    }
    else if (k==0) {
      scorek[0]=-Infinity;
      scorek[1]=prebest[0]+suffbest[lengthsecond]+gap;
    }
    
    if (scorek[0]>=scorek[1])
      tempgapflag=0;
    else
      tempgapflag=1;
    if (scorek[tempgapflag] > maxscore){
      maxscore=scorek[tempgapflag];
      maxindex=k+from2-1;
      gapflag=tempgapflag;
    }
  }
  if (gapflag==0) {
    assert(maxindex>=0);
    assert(imagefirst[m]==-1);
    //      cout << " ### " << m-1 << " -->> " << maxindex-1 << endl;
    imagefirst[m]=maxindex;
    assert(imagesecond[imagefirst[m]]==-1);
    imagesecond[imagefirst[m]]=m;
  }
   else imagefirst[m]=-1;
  
  if (m-1>=from1 && from2 <=maxindex-1+gapflag)
    findmapPAM(seq1, seq2, from1, m-1, from2, maxindex-1+gapflag, matrixIdx, imagefirst, imagesecond, col1, col2, prebest, suffbest);
  if (m+1 <= to1 && maxindex+1 <=to2)
    findmapPAM(seq1, seq2, m+1, to1, maxindex+1, to2, matrixIdx, imagefirst, imagesecond, col1, col2, prebest, suffbest);
  return maxscore;
  
}



double memalignPAM(int *first, int *second, int fromfirst, int tofirst, int fromsecond, 
		   int tosecond, int matrixIdx, int *imagefirst, int *imagesecond){
  assert(fromfirst<=tofirst);
  assert(fromsecond<=tosecond);

  int i;
  for (i=fromfirst;  i<=tofirst;  ++i) assert(first[i]  >= 0 && first[i]  <= 20);
  for (i=fromsecond; i<=tosecond; ++i) assert(second[i] >= 0 && second[i] <= 20);
  
  // Allocate Memory
  double score;
  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;
  double *col1 = new double[lengthfirst+1]; arrayZero(col1, lengthfirst+1);
  double *col2 = new double[lengthfirst+1]; arrayZero(col2, lengthfirst+1);
  double *prebest = new double[lengthsecond+1]; arrayZero(prebest, lengthsecond+1);
  double *suffbest = new double[lengthsecond+1]; arrayZero(suffbest, lengthsecond+1);
  
  for (i=0; i<lengthfirst; ++i)
    imagefirst[i]=-1;
  for (i=0; i<lengthsecond; ++i)
    imagesecond[i]=-1;

  score=findmapPAM(first+fromfirst, second+fromsecond, 0, tofirst-fromfirst, 0, tosecond-fromsecond,
		   matrixIdx, imagefirst, imagesecond, col1, col2, prebest, suffbest);
  
  // Delete memory
  delete[] col1;
  delete[] col2;
  delete[] prebest;
  delete[] suffbest;
  
  return score;
}


void alignExonPair(FilterSequence *hseq, FilterSequence *mseq, int hstart, int hstop, int mstart, int mstop,
		   int hfr, int mfr, int *himg, int *mimg,
		   char *haa, char *maa, int &haastart, int &maastart, int &haastop, int &maastop, int *haaimg, int *maaimg) {

  int i,j;

  //  assert(hseq->get(hstart-2) == BASE_A && hseq->get(hstart-1) == BASE_G && hseq->get(hstop+1) == BASE_G && hseq->get(hstop+2) == BASE_T);
  //  assert(mseq->get(mstart-2) == BASE_A && mseq->get(mstart-1) == BASE_G && mseq->get(mstop+1) == BASE_G && mseq->get(mstop+2) == BASE_T);

  haastart = hstart + (3 - (hstart + 3 - hfr)%3) % 3;
  maastart = mstart + (3 - (mstart + 3 - mfr)%3) % 3;

  assert(haastart%3 == hfr && maastart%3 == mfr);

  haastop = haastart+2; for (; haastop + 3 <= hstop; haastop += 3);
  maastop = maastart+2; for (; maastop + 3 <= mstop; maastop += 3);
  
  assert(! ((haastop-haastart+1)%3) && !((maastop-maastart+1)%3));
  int haalen = (haastop-haastart+1)/3, maalen = (maastop-maastart+1)/3;
  assert(haalen > 2 && maalen > 2);

  char *hnuc = new char[haalen*3]; arrayZero(hnuc, haalen*3);
  char *mnuc = new char[maalen*3]; arrayZero(mnuc, maalen*3);

  for (i=haastart; i<=haastop; ++i) hnuc[i-haastart] = b2c(hseq->get(i));
  for (i=maastart; i<=maastop; ++i) mnuc[i-maastart] = b2c(mseq->get(i));

  Dictionary::translateDNAtoAA(hnuc, haalen, haa);
  Dictionary::translateDNAtoAA(mnuc, maalen, maa);

  int *haaint = new int[haalen]; for (i=0; i<haalen; ++i) haaint[i] = aa_char_num[haa[i]];
  int *maaint = new int[maalen]; for (i=0; i<maalen; ++i) maaint[i] = aa_char_num[maa[i]];

  arrayInit(-2,haaimg,haalen);
  arrayInit(-2,maaimg,maalen);


  // Step 1 in finding map: fix all amino acids that are preserved;
  if (1) // Don't fix anything !!
    for (i=0; i<haalen; ++i) {
      int himg_i = himg[haastart + 3*i];
      if (himg_i <= 0 || himg_i%3 != mfr || himg_i < maastart || himg_i > maastop-2)        continue;
      if (himg[haastart + 3*i + 1] != himg_i + 1 || himg[haastart + 3*i + 2] != himg_i + 2) continue;
      
      int m_i = (himg_i - maastart)/3;
      if (haa[i] == maa[m_i]) {
	haaimg[i]   = m_i;
	maaimg[m_i] = i;
      }
    }

  int hfixed = 0, mfixed = 0;
  for (i=0; i<haalen; ++i) if (haaimg[i] != -2) hfixed++;
  for (i=0; i<maalen; ++i) if (maaimg[i] != -2) mfixed++;

  // Step 2 in finding map: if i and i+2 are fixed, and i+1 can only either mismatch, or gap, then fix i+1;

  
  for (i=1; i<haalen-1; ++i) {
    if (haaimg[i] != -2 || haaimg[i-1] < 0 || haaimg[i+1] < 0) continue;
    if (haaimg[i+1] == haaimg[i-1]+2) {
      haaimg[i]         = haaimg[i-1]+1;
      assert(maaimg[haaimg[i]] == -2);
      maaimg[haaimg[i]] = i;
    }
  }
  for (i=1; i<haalen; ++i)
    if (haaimg[i-1] >= 0 && haaimg[i] > haaimg[i-1]+1)
      for (j = haaimg[i-1]+1; j<haaimg[i]; ++j) maaimg[j] = -1;
  for (i=1; i<maalen; ++i)
    if (maaimg[i-1] >= 0 && maaimg[i] > maaimg[i-1]+1)
      for (j = maaimg[i-1]+1; j<maaimg[i]; ++j) haaimg[j] = -1;

  if (haaimg[0] == -2 && haaimg[1] == 1) {
    haaimg[0] = 0; maaimg[0] = 0;
  }
  if (haaimg[haalen-1] == -2 && haaimg[haalen-2] == maalen-2) {
    haaimg[haalen-1] = maalen-1; maaimg[maalen-1] = haalen-1;
  }


  hfixed = mfixed = 0;
  for (i=0; i<haalen; ++i) if (haaimg[i] != -2) hfixed++;
  for (i=0; i<maalen; ++i) if (maaimg[i] != -2) mfixed++;

  if (haaimg[0] > 0) for (j=0; j<haaimg[0]; ++j) maaimg[j] = -1;
  if (maaimg[0] > 0) for (j=0; j<maaimg[0]; ++j) haaimg[j] = -1;

  if (maaimg[maalen-1]>=0 && maaimg[maalen-1] < haalen-1) for (j = maaimg[maalen-1]+1; j<haalen; ++j) haaimg[j] = -1;
  if (haaimg[haalen-1]>=0 && haaimg[haalen-1] < maalen-1) for (j = haaimg[haalen-1]+1; j<maalen; ++j) maaimg[j] = -1;

  hfixed = mfixed = 0;
  for (i=0; i<haalen; ++i) if (haaimg[i] != -2) hfixed++;
  for (i=0; i<maalen; ++i) if (maaimg[i] != -2) mfixed++;

  // Step 3: aligning the regions in between;
  i = 0; j = 0;
  while (i < haalen && j < maalen) {

    while (i<haalen && haaimg[i] != -2) i++;
    while (j<maalen && maaimg[j] != -2) j++;

    int hregstart = i;
    int mregstart = j;
    
    if (i == haalen || j == maalen) {
      assert(i == haalen && j == maalen);
      break;
    }
	
    while (i<haalen && haaimg[i] == -2) i++;
    while (j<maalen && maaimg[j] == -2) j++;
    
    if (i < haalen && haaimg[i] >= 0) assert(haaimg[i] == j && maaimg[j] == i);

    int hregstop = i-1;
    int mregstop = j-1;

    assert(hregstop - hregstart + 1 > 0 && mregstop - mregstart + 1 > 0 && (hregstop-hregstart + mregstop-mregstart + 2) >= 3);

    memalignPAM(haaint, maaint, hregstart, hregstop, mregstart, mregstop, 1, haaimg + hregstart, maaimg + mregstart);
    for (j=hregstart; j<=hregstop; ++j) if (haaimg[j] >= 0) haaimg[j] += mregstart;
    for (j=mregstart; j<=mregstop; ++j) if (maaimg[j] >= 0) maaimg[j] += hregstart;
  }
  
  for (i=0; i<haalen; ++i) {
    assert(haaimg[i] >= -1);
    if (haaimg[i] >= 0) assert(maaimg[haaimg[i]]==i);
  }
  for (i=0; i<maalen; ++i) assert(maaimg[i] >= -1);
  
  delete[] hnuc;
  delete[] mnuc;
  delete[] haaint;
  delete[] maaint;
}



int theexonImagesToTry(FilterSequence *hseq, FilterSequence *mseq, int hstart, int hstop, int *himg, int *mimg,
		     int *mstarts, int *mstops) {
  
  int i,j;

  int agmaps=0, gtmaps=0;
  
  if (himg[hstart-2] > 0 && himg[hstart-1] > 0 && mseq->get(himg[hstart-2]) == BASE_A && mseq->get(himg[hstart-1]) == BASE_G
      && himg[hstart-1] == himg[hstart-2]+1) agmaps = 1;
  if (himg[hstop +1] > 0 && himg[hstop +2] > 0 && mseq->get(himg[hstop +1]) == BASE_G && mseq->get(himg[hstop +2]) == BASE_T
      && himg[hstop +1] == himg[hstop+2] -1) gtmaps = 1;
  
  int mstart1=-1, mstart2=-1, mstop1=-1, mstop2=-1;

  if (agmaps) mstart1 = himg[hstart-1]+1;
  else {
    for (i = hstart; i > 0 && himg[i] <= 0; i--);
    if (i>0) {
      i = himg[i];
      for (; i>0 && (mseq->get(i) != BASE_A || mseq->get(i+1) != BASE_G); --i);
      if (i>0) mstart1 = i+2;
    }
    
    for (i = hstart+1; i<hseq->get_length()-1 && himg[i] <= 0; ++i);
    if (i<hseq->get_length()-2) {
      i = himg[i];
      for (; i<mseq->get_length()-2 && (mseq->get(i) != BASE_A || mseq->get(i+1) != BASE_G); ++i);
      if (i<mseq->get_length()-2) mstart2 = i+2;
    }
  }

  if (gtmaps) mstop1 = himg[hstop+1]-1;
  else {
    for (i = hstop; i > 0 && himg[i] <= 0; i--);
    if (i>0) {
      i = himg[i];
      for (; i>0 && (mseq->get(i) != BASE_G || mseq->get(i+1) != BASE_T); --i);
      if (i>0) mstop1 = i-1;
    }

    for (i = hstop+1; i<hseq->get_length()-1 && himg[i] <= 0; ++i);
    if (i<hseq->get_length()-1) {
      i = himg[i];
      for (; i<mseq->get_length()-1 && (mseq->get(i) != BASE_G || mseq->get(i+1) != BASE_T); ++i);
      if (i<mseq->get_length()-1) mstop2 = i-1;
    }
  }

  if (mstart1 != -1 && mstop1 != -1) {
    assert(mseq->get(mstart1-2) == BASE_A);
    assert(mseq->get(mstart1-1) == BASE_G);
    assert(mseq->get(mstop1 +1) == BASE_G);
    assert(mseq->get(mstop1 +2) == BASE_T);
  }

  if ( (mstart1 == -1 && mstart2 == -1) || (mstop1 == -1 && mstop2 == -1) ) return 0;

  if (mstart1 == -1) { mstart1 = mstart2; mstart2 = -1; }
  if (mstop1  == -1) { mstop1  = mstop2;  mstop2  = -1; }


  mstarts[0] = mstart1;
  mstops[0]  = mstop1;
  int exonCount = 1;
  if (mstop2 != -1) {
    mstarts[1] = mstart1;
    mstops[1]  = mstop2;
    exonCount++;
    if (mstart2 != -1) {
      mstarts[2] = mstarts[3] = mstart2;
      mstops[2]  = mstop1;
      mstops[3]  = mstop2;
      exonCount += 2;
    }
  }
  else if (mstart2 != -1) {
    mstarts[1] = mstart2;
    mstops[1]  = mstop1;
    exonCount++;
  }

  int changed = 0;
  do {
    changed = 0;
    for (i=0; i<exonCount; ++i)
      if (mstops[i] - mstarts[i] + 1 < 9) {
	for (j=i+1; j<exonCount; ++j) {
	  mstarts[j-1] = mstarts[j];
	  mstops[ j-1] = mstops[ j];
	}
	exonCount--;
	changed = 1;
      }
  } while (changed);

  for (i=0; i<exonCount; ++i) {
    assert(mseq->get(mstarts[i]-2) == BASE_A);
    assert(mseq->get(mstarts[i]-1) == BASE_G);
    assert(mseq->get(mstops[i] +1) == BASE_G);
    assert(mseq->get(mstops[i] +2) == BASE_T);
  }
  return exonCount;
}

